function [nrmGrad, dirGrad, gradIm, im] = getGradNrmDir(im, params, fHandle)
  % [nrmGrad, dirGrad, gradIm] = getGradNrmDir(im, params, fHandle)
  % Assumes im is double, either RGB (ny x nx x 3) or 
  % greylevel (ny x nx).  If RGB, uses the average of the channels for
  % a brightness image.
  % params.useGauss (defaults to true) false uses dx.m dy.m for deriv.
  % params.sigma  (defaults to 1.2) stdev of gaussian used when
  %               useGauss is set to true.
  % params.superSamp (defaults to 1).  If > 1, then supersample the 
  %   brightness image using linear interp.
  % fHandle (optional) A figure handle to show a colour coded
  %          gradient direction, e.g. pass the argument: figure(100) 
  % Returns
  %  nrmGrad 
  if exist('fHandle', 'var')
    CHATTY = true;
  else
    CHATTY = false;
  end
  if ~exist('params', 'var')
    params.useGauss = true;
    params.sigma = 1.2;
    params.superSample = 1;
  end
  if params.useGauss && ~isfield(params, 'sigma')
    params.sigma = 1.2;
  end
  if ~isfield(params, 'superSample')
    params.superSample = 1;
  else
    params.superSample = round(params.superSample);
  end
  
  if length(size(im)) == 3
    im = sum(im,3)/size(im,3);
  end
  szIm = size(im);
  
  if params.superSample > 1
    superSamp = params.superSample;
    xi = 1:(1.0/superSamp):szIm(2);
    yi = 1:(1.0/superSamp):szIm(1);
    im = interp2(im(:,:), xi, yi');
    szIm = size(im);
  end

  if params.useGauss
    % Gradient image
    sigmaGrad = params.sigma;
    sigmaSqr = sigmaGrad*sigmaGrad;
    gGradSize = 2 * round(3.0 * sigmaGrad) + 1;
    x = [1:gGradSize] - round((gGradSize+1)/2);
    gGrad = exp(- x .* x / (2.0*sigmaSqr));
    gGrad = gGrad/ sum(gGrad(:));
    gxGrad = - (x / sigmaSqr) .* gGrad;

    gradIm = zeros(size(im,1), size(im,2), 2);

    tmp = upConv(im, gxGrad, 'repeat');
    gradIm(:,:,1) = upConv(tmp, gGrad', 'repeat');
    tmp = upConv(im, gGrad, 'repeat');
    gradIm(:,:,2) = upConv(tmp, gxGrad', 'repeat');
  else
    gradIm = zeros([size(im), 2]);
    gradIm(:,:,1) = dx(im, 'repeat');
    gradIm(:,:,2) = dy(im, 'repeat');
  end

  % Norm of gradient image
  nrmGrad = sqrt(sum(gradIm(:,:,1:2).^2, 3));
  
  % Direction of gradient image
  dirGrad = atan2(gradIm(:,:,2), gradIm(:,:,1))/pi;
  
  if CHATTY
    clabel = uint8(round((dirGrad+1)/2*255));
    cmap = colormap(hsv(256));
    gradRGB = reshape(cmap(clabel(:)+1,:), [size(im), 3]);
    gradRGB = gradRGB .* repmat((nrmGrad/max(nrmGrad(:))).^0.5, [1,1,3]);
    figure(fHandle); clf; 
    image(gradRGB);
    axis equal;
  end
  return; 