function [hist, thetaCenters] = getAlignedHOG(z, nrmGradIm, dirGradIm, ...
                                              superSamp, paramsHoG, hFig)
%  [hist, thetaCenters] = getAlignedHOG(z, nrmGradIm, dirGradIm, ...
%                            paramsHoG, hFig)
%  Input 
%    z a 2 x K array of center points to apply HoG
%    nrmGradIm the 2-norm of the image gradient
%    dirGradIm the image of gradient directions, where
%      dirGradIm = atan2( gradIy, gradIx )/pi has values in [-1 1]
%    superSamp an integer >= 1 indicating the superSampling rate of
%      the nrmGradIm and dirGradIm.  The integral images are downsampled
%      by superSample before computing the HoG cell responses.
%    paramsHoG  HoG params struct
%    [hFig] figure handle for intermediate results
%  Output
%    hist nHist x K matrix of HoG responses.
%         HoG responses stored in column of nTheta distinct orientations
%         for the first cell, then second cell, and so on.
%         nHist = nTheta * nCell

  if exist('hFig', 'var')
    display = true;
  else
    display = false;
  end
  
  maxAmp = paramsHoG.maxAmp;
  minAmp = paramsHoG.minAmp;
  szIm = size(nrmGradIm);
  
  
  nSamp = size(z,2);
  nCell = length(paramsHoG.cellBox);
  nTheta = paramsHoG.nTheta;
  
  % Initialize histograms to zero
  hist = zeros([nTheta, nCell, nSamp]);
  
  % Compute direction bin labels
  [labelIm, thetaCenters] = labelDir(dirGradIm, nTheta, paramsHoG.wrapOrient);
  
  % Saturate the gradient amplitude at the maximum binLAmp(end)
  nrmGradIm = min(nrmGradIm, paramsHoG.maxAmp);
  
  % Zero out small gradient amplitudes
  idx = nrmGradIm < paramsHoG.minAmp;
  nrmGradIm(idx) = 0;
  
  if display

    % Quantized direction of gradient image
    cmap = [0 0 0; hsv(nTheta)];
    gradRGB = reshape(cmap(labelIm(:)+1,:), [szIm 3]);
    gradRGB = gradRGB .* repmat(nrmGradIm, [1,1,3])/paramsHoG.maxAmp;
    figure(hFig); clf; 
    image(gradRGB); axis equal;
    hold on;
    plot(z(1,:), z(2,:), '*w');
    pause(0.1);
    
  end

  % Pad the gradient norm and label images with a row 
  % and column of zeros.  This is useful for the use 
  % of integral images.
  padLabel = [zeros(size(labelIm,1),1) labelIm];
  padLabel = [zeros(1,size(padLabel,2)) ; padLabel];
  padNrmGrad =[zeros(size(nrmGradIm,1),1) nrmGradIm];
  padNrmGrad = [zeros(1,size(padNrmGrad,2)) ; padNrmGrad];
  
  % Build the integral images for each gradient orientation range.
  for kLabel = 1:nTheta
    
    
    % Get the integral image for this label
    intIm = integralIm(padNrmGrad.*(padLabel == kLabel));
    if superSamp > 1
      % Downsample intIm
      szIntIm = size(intIm);
      intIm = intIm(1:superSamp:szIntIm(1), 1:superSamp:szIntIm(2));
      szIntIm = size(intIm);
    else
      szIntIm = size(intIm);
    end
    intIm = intIm(:);
    
    % Extract the histogram bins for this orientation label.
    for ic = 1:nCell
      
      % Collect histogram within bbox
      bbox = paramsHoG.cellBox(ic).bbox;

      % Compute [x0,y0] one pixel up and to the
      % left of the bounding box, in the padded integral
      % image
      x0 = round(z(1,:) + bbox(1,1));
      y0 = round(z(2,:) + bbox(1,2));
      % Compute [x1,y1] at the bottom right corner of the
      % bounding box, in the padded integral image.
      x1 = round(z(1,:) + (bbox(2,1)+1));
      y1 = round(z(2,:) + (bbox(2,2)+1));
      
      % Clip to boundaries of image
      x0 = max(x0, 1);
      y0 = max(y0, 1);
      x1 = max(x0, x1);
      y1 = max(y0, y1);
      
      x1 = min(x1, szIntIm(2));
      y1 = min(y1, szIntIm(1));
      x0 = min(x0, x1);
      y0 = min(y0, y1);
      
      
      if  display &  kLabel == 1 
        for k=1:length(x0)
          % Conversion of rounded box back to 
          % image coords
          lBox = [x0(k) y0(k); x1(k)-1, y1(k)-1];
          figure(hFig); hold on;
          plot([lBox(1,1) lBox(2,1)], [lBox(1,2) lBox(1,2)],'r');
          plot([lBox(1,1) lBox(2,1)], [lBox(2,2) lBox(2,2)],'r');
          plot([lBox(1,1) lBox(1,1)], [lBox(1,2) lBox(2,2)],'r');
          plot([lBox(2,1) lBox(2,1)], [lBox(1,2) lBox(2,2)],'r');
        end

      end
      
      % Build values within bounding boxes for all sample points
      val = intIm( y1 + szIntIm(1)* (x1-1));
      val = val - intIm(y0 + szIntIm(1) * (x1 - 1));
      val = val - intIm(y1 + szIntIm(1) * (x0-1));
      val = val + intIm(y0 + szIntIm(1) * (x0 - 1));
      hist(kLabel, ic, :) = val;
      
    end % over ic
  end  % over kLabel

  hist = reshape(hist, nTheta*nCell, nSamp);
  
  hist = max(hist, 0.0);
  
  return;
  
