function [r] = groundCoordVal(X, Y, checkSize, checksPerSide, groundGray)
  % Generate point samples of an X,Y plane containing a checkerboard.
  % Input:  
  %   World coordinates [X; Y; 0], where X and Y are 1 by P vectors,
  %   representing points on the ground plane (containing the
  %   checkerboard).  You can think of X and Y as in centimeters.
  %   The center of the checkerboard is at X=0, Y = 0.
  %   checkSize - size of one check in centimeters
  %   checksPerSide - number of checks per side on checkerboard (must
  %                   be an even integer)
  %   groundGray - the graylevel of the groundplane away from the
  %                checkerboard.
  % Returns
  %   r a 1 x P matrix with r(k) equal to the gray level of the
  %     checkerboard plane at (X(k), Y(k), 0)^T.
  
  % Reshape X,Y to be column vectors 
  X = X(:);
  Y = Y(:);
  r = zeros(size(X));
  groundGray = min(max(0, groundGray), 255);
  
  % Decide whether plane point (X, Y)  is white or black.
  cx = floor(X/checkSize);
  cy = floor(Y/checkSize);
  bitFlip = rem(cx+cy,2) ~= 0;
    
  % Fill in white checks
  r(bitFlip) = 255;
  % Black checks are already set to zero.

  % Set points on groundplane outside checkerboard to groundGray.
  checkRad = round(checksPerSide/2);
  
  r((max(cx,cy) > checkRad - 1) | (min(cx,cy) < -checkRad)) = groundGray;

  % Transpose r to be a row vector.
  r = r';