/* Server for cloud stuff      *\
|*                             *|
\*                             */

#include <iostream>
//#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/types.h> 
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <unordered_map>
#include <unordered_set>
#include <fstream>
#include <ctime>

#include "cloud_protocol.h"
#include "cloud_utils.h"

#define TIMEOUT 30


// store the newest CRC's we have, based on file name
std::unordered_map<std::string, uint64_t> newest_crcs;
// working directory
std::string cwd;

// return full patch to file/patch
std::string path_to_file(std::string fname, bool curr=true, uint64_t crc=0){
  std::string out_string = cwd + "/" + fname + "/";
  if(curr){
    out_string += "curr";
  }
  else{
    out_string += std::to_string(crc);
  }
  return out_string;
}

// check if a file exists or not
bool file_exists(std::string fname, bool curr=true, uint64_t crc=0){

  struct stat tmp_stat;
  return stat(path_to_file(fname,curr,crc).c_str(),&tmp_stat) == 0;
}

// initialization
int init(){
  //change working dirs
  const char* wd = "./.cloudserver";
  //make the dir just in case
  mkdir(wd, 0755);
  if(chdir(wd) < 0){
    std::cout<<"Error with creating/changing directories!"<<std::endl;
    return -1;
  }

  //store our working directory
  cwd = std::string(get_current_dir_name());

  //read newest CRC's from file, if it exists
  //example contents:
  //file.txt|777777|file2.txt|8888
  std::ifstream crc_if;
  crc_if.open(".crcfile");
  if(crc_if.is_open()){
    //add each entry to newest_crcs
    std::string name_in;
    while(std::getline(crc_if, name_in,'|')){
      if(name_in.empty() || crc_if.eof()){
        break;
      }
      std::string tmp_num;
      std::getline(crc_if, tmp_num,'|');
      std::cout<<"."<<std::flush;
      uint32_t num_in = stoul(tmp_num, nullptr, 0);

      newest_crcs.emplace(name_in, num_in);
    }
    crc_if.close();
  }

  return 0;
}

int main(int argc, char *argv[])
{

  std::cout<< "Starting server."<<std::flush;

  //initialize stuff
  if(init() < 0){
    std::cout<<".error initializing local stuff!"<<std::endl;
    return -1;
  }
  
  std::cout<<"."<<std::flush;

  // open socket, keep track of where
  int sock_id = socket(AF_INET, SOCK_STREAM, 0);
  if(sock_id < 0){
    std::cout<<".error opening socket!"<<std::endl;
    return -1;
  }
  
  std::cout<<"."<<std::flush;

  // the sockaddr corresponding to opening socket
  struct sockaddr_in addr_in;
  bzero((char*) &addr_in, sizeof(addr_in));
  addr_in.sin_family = AF_INET;
  addr_in.sin_addr.s_addr = INADDR_ANY;
  addr_in.sin_port = htons(SERVER_PORT);

  // try binding socket
  if(bind(sock_id, (struct sockaddr*) &addr_in, sizeof(addr_in)) < 0){
    std::cout<<".error binding socket!"<<std::endl;
    return -1;
  }
  std::cout<<".done!"<<std::endl;


  //now we enter main loop
  while(true){
  
    std::cout<<"Waiting for connection.."<<std::flush;

    //listen for incoming connections
    listen(sock_id, 5);

    //try to accept incoming connection
    struct sockaddr_in addr_conn;
    unsigned int conn_size = sizeof(addr_conn);
    int sock_conn_id = accept(sock_id, (struct sockaddr*) &addr_conn, &conn_size);
    std::cout<<"."<<std::flush;

    if (sock_conn_id < 0){
      std::cout<<".error accepting incoming connection!"<<std::endl;
      return -1;
    }
    std::cout<<"."<<std::flush;

    std::cout << ".connected!" << std::endl;

    //if we are here, we are happily connected!
    time_t time_rec = time(NULL);
    time_t time_now = time(NULL);
    while(difftime(time(NULL),time_rec) < TIMEOUT){
      time_now = time(NULL);
      struct tm* time_tmp;
      time_tmp = localtime(&time_now);
      std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Waiting for message." << std::endl;
      //get the header info
      struct cloud_hdr cloud_in;
      if(read_x_bytes(sock_conn_id, sizeof(cloud_hdr_t), (uint8_t*) &cloud_in) < (signed int) sizeof(cloud_hdr_t)){
        std::cout<<"Connection closed / incomplete message!"<<std::endl;
        break;
      }

      time_now = time(NULL);
      time_tmp = localtime(&time_now);
      std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Received a message of type: " << (int)cloud_in.cloud_reqtype << std::endl;

      //now let's see what we do based on what the header is
      switch(cloud_in.cloud_reqtype){

        case req_update:
          {
            //let's add the patch!
            //get fname
            char* fname_c = new char[cloud_in.cloud_fnamelen];
            if(read_x_bytes(sock_conn_id, cloud_in.cloud_fnamelen, (uint8_t*) fname_c) < 0){
              std::cout<<"Error while reading transmission!"<<std::endl;
              continue;
            }
            std::string fname(fname_c, cloud_in.cloud_fnamelen);
            delete[] fname_c;

            //first let's check if this patch is compatible, if not we will just req full guy
            if(newest_crcs.at(fname) != cloud_in.cloud_oldcrc){
              //send a request for the full file!
              struct cloud_hdr req_out;
              req_out.cloud_reqtype = req_updatefull;
              req_out.cloud_userid = 0;
              req_out.cloud_fnamelen = cloud_in.cloud_fnamelen;
              req_out.cloud_datalen = 0;
              req_out.cloud_oldcrc = 0;
              req_out.cloud_newcrc = 0;
              uint8_t* out_buf = new uint8_t[sizeof(cloud_hdr_t) + req_out.cloud_fnamelen];
              memcpy(out_buf, (uint8_t*) &req_out, sizeof(cloud_hdr_t));
              memcpy(out_buf+sizeof(cloud_hdr_t), fname.c_str(), req_out.cloud_fnamelen);
              
              time_now = time(NULL);
              time_tmp = localtime(&time_now);
              std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Sending a req_updatefull for " << fname <<"!" << std::endl;
              if(write_x_bytes(sock_conn_id, sizeof(cloud_hdr_t) + req_out.cloud_fnamelen, out_buf) < 0){
                std::cout<<"Error while sending transmission!"<<std::endl;
                return -1;
              }
              delete[] out_buf;
            }

            //save patch to file
            char* buf = new char[cloud_in.cloud_datalen];
            if(read_x_bytes(sock_conn_id, cloud_in.cloud_datalen, (uint8_t*) buf) < 0){
              std::cout<<"Error while reading transmission!"<<std::endl;
              continue;
            }
            //save it as oldcrc, ie. the patch needed to get from oldcrc to curr
            std::ofstream patch_out(path_to_file(fname,false,cloud_in.cloud_oldcrc).c_str(), std::ios::out | std::ios::trunc);
            if(!patch_out.is_open()){
              std::cout<<"Error creating patch file!"<<std::endl;
            }
            patch_out.write(buf, cloud_in.cloud_datalen);
            patch_out.close();

            //apply patch to file
	        std::string tmp_cmd = "patch " + path_to_file(fname) + " < " + path_to_file(fname,false,cloud_in.cloud_oldcrc);
	        
	        time_now = time(NULL);
            time_tmp = localtime(&time_now);
            std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Applying patch with command: " << tmp_cmd << std::endl;
            system(tmp_cmd.c_str());

            //update other patches with this patch
            DIR* this_dir;
            struct dirent* this_dirent;
	        std::string tmp_dir = cwd + "/" + fname + "/";

            this_dir = opendir(tmp_dir.c_str());
            if(this_dir == nullptr){
              std::cout << "Error iterating through files!"<<std::endl;
              return -1;
            }
            while((this_dirent = readdir(this_dir))){
              if(this_dirent->d_type == DT_REG && strcmp(this_dirent->d_name,"curr")!= 0 && strcmp(this_dirent->d_name,std::to_string(cloud_in.cloud_oldcrc).c_str()) != 0){
		        std::string tmp_cmd = "combinediff " + tmp_dir + std::string(this_dirent->d_name) + " ";
		        tmp_cmd += tmp_dir + std::to_string(cloud_in.cloud_oldcrc) + " > " + tmp_dir + std::string(this_dirent->d_name);
		        tmp_cmd += "t && mv " + tmp_dir + std::string(this_dirent->d_name) + "t " + tmp_dir + std::string(this_dirent->d_name);
		        
		        
	            time_now = time(NULL);
                time_tmp = localtime(&time_now);
		        std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Combining patches with command: " << tmp_cmd << std::endl;
		        
                system(tmp_cmd.c_str());
                // TODO: check if correct!
              }
            }

            //update crc in newest_crcs
            newest_crcs.at(fname) = cloud_in.cloud_newcrc;
            break;
          }



        case req_updatefull:
          {
            //let's add the file!
            //change directories
            char* fname_c = new char[cloud_in.cloud_fnamelen];
            if(read_x_bytes(sock_conn_id, cloud_in.cloud_fnamelen, (uint8_t*) fname_c) < 0){
              std::cout<<"Error while reading transmission!"<<std::endl;
              continue;
            }
            std::string fname(fname_c, cloud_in.cloud_fnamelen);
            delete[] fname_c;

            std::string old_dir(get_current_dir_name());
            std::string new_dir = old_dir + "/" + fname;
            //make the dir just in case
            mkdir(new_dir.c_str(), 0755);
           // chdir(new_dir.c_str());

            //save file
            char* buf = new char[cloud_in.cloud_datalen];
            if(read_x_bytes(sock_conn_id, cloud_in.cloud_datalen, (uint8_t*) buf) < 0){
              std::cout<<"Error while reading transmission!"<<std::endl;
              continue;
            }
            //check if curr exists
            if(file_exists(fname, true)){
              //curr already exists!
              //save new file as new_curr
              std::string tmp(cwd);
              tmp += "/" + fname + "/new_curr";
              std::ofstream new_curr_out(tmp.c_str(), std::ios::out | std::ios::trunc);
              //std::ofstream new_curr_out(path_to_file(fname).c_str(), std::ios::out | std::ios::trunc);
              if(!new_curr_out.is_open()){
                std::cout<<"Error creating new_curr file!"<<std::endl;
              }
              new_curr_out.write(buf, cloud_in.cloud_datalen);
              new_curr_out.close();
              //create patch file to get from curr > new_curr
              std::string diff_str = "diff -u --label=curr --label=curr " + path_to_file(fname) + " ";
              diff_str += cwd + "/" + fname + "/new_curr > ";
              diff_str += path_to_file(fname, false, newest_crcs.at(fname));
              
	          time_now = time(NULL);
              time_tmp = localtime(&time_now);
              std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Creating patch with command: " << diff_str << std::endl;
              system(diff_str.c_str());

              //for each file in the folder that's not curr or curr_new, combinediff w `newest_crcs.at(fname)`
              DIR* this_dir;
              struct dirent* this_dirent;
	          std::string tmp_dir = cwd + "/" + fname + "/"; 
              this_dir = opendir(tmp_dir.c_str());
              if(this_dir == nullptr){
                std::cout << "Error iterating through files!"<<std::endl;
                return -1;
              }

              while((this_dirent = readdir(this_dir))){
                if(this_dirent->d_type == DT_REG && strcmp(this_dirent->d_name,"curr")!= 0 && strcmp(this_dirent->d_name,"new_curr")!= 0 && strcmp(this_dirent->d_name, std::to_string(newest_crcs.at(fname)).c_str()) != 0){
		          std::string tmp_cmd = "combinediff " + tmp_dir + std::string(this_dirent->d_name) + " ";
		          tmp_cmd += path_to_file(fname, false, newest_crcs.at(fname)) + " > " + tmp_dir + std::string(this_dirent->d_name);
		          tmp_cmd += "t && mv " + tmp_dir + std::string(this_dirent->d_name) + "t " + tmp_dir + std::string(this_dirent->d_name);

	              time_now = time(NULL);
                  time_tmp = localtime(&time_now);
                  std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Combining patches with command: " << tmp_cmd << std::endl;
                  system(tmp_cmd.c_str());
                  // TODO: check if this works / what this calls!!!!
                }
              }

              //we've applied all patches, now we delete curr and rename new_curr to curr
              remove(path_to_file(fname).c_str());
	          std::string tmp_newcurr = tmp_dir + "new_curr";
              rename(tmp_newcurr.c_str(), path_to_file(fname).c_str());
            }
            else{
              //curr does not exist, save file as curr
              std::ofstream curr_out(path_to_file(fname).c_str(), std::ios::out | std::ios::trunc);
              if(!curr_out.is_open()){
                std::cout<<"Error creating curr file!"<<std::endl;
              }
              curr_out.write(buf, cloud_in.cloud_datalen);
              curr_out.close();
            }

            //update crc in newest_crcs
            if(newest_crcs.count(fname) > 0){
              newest_crcs.at(fname) = cloud_in.cloud_newcrc;
            }
            else{
              uint64_t tmp = cloud_in.cloud_newcrc;
              newest_crcs.emplace(fname, tmp);
            }

            break;
          }



        case req_reqfull:
          {
            //let's send the appropriate file!
            char* fname_c = new char[cloud_in.cloud_fnamelen];
            if(read_x_bytes(sock_conn_id, cloud_in.cloud_fnamelen, (uint8_t*) fname_c) < 0){
              std::cout<<"Error while reading transmission!"<<std::endl;
              continue;
            }
            std::string fname(fname_c, cloud_in.cloud_fnamelen);
            delete[] fname_c;

            //does this file exist?
            if(!file_exists(fname,true)){
              std::cout<<"Error finding requested file! Ignoring!"<<std::endl;
              continue;
            }

            struct cloud_hdr cloud_hdr_out;
            cloud_hdr_out.cloud_reqtype = req_updatefull;
            cloud_hdr_out.cloud_userid = 0;
            cloud_hdr_out.cloud_fnamelen = fname.size();
            cloud_hdr_out.cloud_oldcrc = 0;
            cloud_hdr_out.cloud_newcrc = newest_crcs.at(fname);


            std::ifstream file_in(path_to_file(fname).c_str(), std::ios::binary);
            if(!file_in.is_open()){
              std::cout<<"Error accessing file!"<<std::endl;
              continue;
            }
            //find the length of the file
            file_in.seekg (0, file_in.end);
            cloud_hdr_out.cloud_datalen = file_in.tellg();
            file_in.seekg (0, file_in.beg);

            //now we know all the lengths and stuff, let's allocate and fill!
            uint8_t* out_buf = new uint8_t[sizeof(cloud_hdr_t) + cloud_hdr_out.cloud_fnamelen + cloud_hdr_out.cloud_datalen];
            memcpy( out_buf, (uint8_t*) &cloud_hdr_out, sizeof(cloud_hdr_t) );
            memcpy( out_buf + sizeof(cloud_hdr_t), fname.c_str(), cloud_hdr_out.cloud_fnamelen );
            file_in.read((char*)out_buf + sizeof(cloud_hdr_t) + cloud_hdr_out.cloud_fnamelen, cloud_hdr_out.cloud_datalen);
            file_in.close();

            //let's try to send it!
	        time_now = time(NULL);
            time_tmp = localtime(&time_now);
            std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Sending full file " << fname << "!" << std::endl;
            if(write_x_bytes(sock_conn_id, sizeof(cloud_hdr_t) + cloud_hdr_out.cloud_fnamelen + cloud_hdr_out.cloud_datalen, out_buf) < 0){
              std::cout<<"Error while sending transmission!"<<std::endl;
              return -1;
            }

            break;
          }



        case req_reqpatch:
          {
            //let's send the appropriate file!
            char* fname_c = new char[cloud_in.cloud_fnamelen];
            if(read_x_bytes(sock_conn_id, cloud_in.cloud_fnamelen, (uint8_t*) fname_c) < 0){
              std::cout<<"Error while reading transmission!"<<std::endl;
              continue;
            }
            std::string fname(fname_c, cloud_in.cloud_fnamelen);
            delete[] fname_c;

            //does this file exist?
            if(!file_exists(fname,false,cloud_in.cloud_oldcrc)){
              std::cout<<"Error finding requested file! Ignoring!"<<std::endl;
              continue;
            }

            struct cloud_hdr cloud_hdr_out;
            cloud_hdr_out.cloud_reqtype = req_update;
            cloud_hdr_out.cloud_userid = 0;
            cloud_hdr_out.cloud_fnamelen = fname.size();
            cloud_hdr_out.cloud_oldcrc = cloud_in.cloud_oldcrc;
            cloud_hdr_out.cloud_newcrc = newest_crcs.at(fname);


            std::ifstream file_in(path_to_file(fname).c_str(), std::ios::binary);
            if(!file_in.is_open()){
              std::cout<<"Error accessing file!"<<std::endl;
              continue;
            }
            //find the length of the file
            file_in.seekg (0, file_in.end);
            cloud_hdr_out.cloud_datalen = file_in.tellg();
            file_in.seekg (0, file_in.beg);

            //now we know all the lengths and stuff, let's allocate and fill!
            char* out_buf = new char[sizeof(cloud_hdr_t) + cloud_hdr_out.cloud_fnamelen + cloud_hdr_out.cloud_datalen];
            memcpy( out_buf, (char*) &cloud_hdr_out, sizeof(cloud_hdr_t) );
            memcpy( out_buf + sizeof(cloud_hdr_t), fname.c_str(), cloud_hdr_out.cloud_fnamelen );
            file_in.read(out_buf + sizeof(cloud_hdr_t) + cloud_hdr_out.cloud_fnamelen, cloud_hdr_out.cloud_datalen);
            file_in.close();

            //let's try to send it!
	        time_now = time(NULL);
            time_tmp = localtime(&time_now);
            std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Sending patch of file " << fname << "!" << std::endl;
            if(write_x_bytes(sock_conn_id, sizeof(cloud_hdr_t) + cloud_hdr_out.cloud_fnamelen + cloud_hdr_out.cloud_datalen, (uint8_t*)out_buf) < 0){
              std::cout<<"Error while sending transmission!"<<std::endl;
              return -1;
            }

            break;
          }



        case req_checkall:
          {
            //let's check the list of filenames / crc's to ours and respond accordingly!
            int remaining_len = cloud_in.cloud_datalen;
            std::unordered_set<std::string> files_in;
            
            while(remaining_len > 0){
              //for each entry
              uint16_t fname_len;
              uint64_t crc_in;

              //read the data
              if(read_x_bytes(sock_conn_id, 2, (uint8_t*) &fname_len) < 0 ||
                  read_x_bytes(sock_conn_id, 8, (uint8_t*) &crc_in) < 0){
                std::cout<<"Error while reading transmission!"<<std::endl;
                continue;
              }

              char* fname_c = new char[fname_len+1];
              fname_c[fname_len] = 0;
              if(read_x_bytes(sock_conn_id, fname_len, (uint8_t*) fname_c) < 0 ){
                std::cout<<"Error while reading transmission!"<<std::endl;
                continue;
              }
              std::string fname(fname_c);
              delete[] fname_c;

              //add filename to set of files received
              files_in.insert(fname);

              //update remaining length of message
              remaining_len -= (2+8+fname_len);

              //check if we have crc entry
              if(newest_crcs.count(fname) == 0){
                //if we don't, send a reqfull
                struct cloud_hdr out_hdr;
                out_hdr.cloud_reqtype = req_reqfull;
                out_hdr.cloud_userid = 0;
                out_hdr.cloud_oldcrc = 0;
                out_hdr.cloud_newcrc = 0;
                out_hdr.cloud_datalen = 0;
                out_hdr.cloud_fnamelen = fname_len;

                uint8_t* out_buf = new uint8_t[sizeof(cloud_hdr_t) + fname_len];
                memcpy(out_buf, (uint8_t*) &out_hdr, sizeof(cloud_hdr_t));
                memcpy(out_buf + sizeof(cloud_hdr_t), fname.c_str(), fname_len);

	            time_now = time(NULL);
                time_tmp = localtime(&time_now);
                std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Sending a req_reqfull for " << fname << "!" << std::endl;
                if(write_x_bytes(sock_conn_id, sizeof(cloud_hdr_t) + fname_len, out_buf) < 0){
                  std::cout<<"Error while sending transmission!"<<std::endl;
                  return -1;
                }
              }
              else{
                if(newest_crcs.at(fname) == crc_in){
                  //all's good, send uptodate
                  struct cloud_hdr out_hdr;
                  out_hdr.cloud_reqtype = req_uptodate;
                  out_hdr.cloud_userid = 0;
                  out_hdr.cloud_oldcrc = 0;
                  out_hdr.cloud_newcrc = crc_in;
                  out_hdr.cloud_datalen = 0;
                  out_hdr.cloud_fnamelen = fname_len;

                  uint8_t* out_buf = new uint8_t[sizeof(cloud_hdr_t) + fname_len];
                  memcpy(out_buf, (uint8_t*) &out_hdr, sizeof(cloud_hdr_t));
                  memcpy(out_buf + sizeof(cloud_hdr_t), fname.c_str(), fname_len);

	              time_now = time(NULL);
                  time_tmp = localtime(&time_now);
                  std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Sending a req_uptodate for " << fname << "!" << std::endl;
                  if(write_x_bytes(sock_conn_id, sizeof(cloud_hdr_t) + fname_len, out_buf) < 0){
                    std::cout<<"Error while sending transmission!"<<std::endl;
                    return -1;
                  }
                }
                else{
                  //so we have the file but the crc's dont match up! check if we have a patch
                  if(file_exists(fname,false,crc_in)){
                    //send patch!
                    struct cloud_hdr out_hdr;
                    out_hdr.cloud_reqtype = req_update;
                    out_hdr.cloud_userid = 0;
                    out_hdr.cloud_oldcrc = crc_in;
                    out_hdr.cloud_newcrc = newest_crcs.at(fname);
                    out_hdr.cloud_fnamelen = fname_len;


                    std::ifstream file_in(path_to_file(fname,false,crc_in).c_str(), std::ios::binary);
                    if(!file_in.is_open()){
                      std::cout<<"Error accessing file!"<<std::endl;
                      continue;
                    }

                    //find the length of the patch
                    file_in.seekg (0, file_in.end);
                    out_hdr.cloud_datalen = file_in.tellg();
                    file_in.seekg (0, file_in.beg);

                    //now we know all the lengths and stuff, let's allocate and fill!
                    char* out_buf = new char[sizeof(cloud_hdr_t) + out_hdr.cloud_fnamelen + out_hdr.cloud_datalen];
                    memcpy( out_buf, (char*) &out_hdr, sizeof(cloud_hdr_t) );
                    memcpy( out_buf + sizeof(cloud_hdr_t), fname.c_str(), out_hdr.cloud_fnamelen );
                    file_in.read(out_buf + sizeof(cloud_hdr_t) + out_hdr.cloud_fnamelen, out_hdr.cloud_datalen);
                    file_in.close();

	                time_now = time(NULL);
                    time_tmp = localtime(&time_now);
                    std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Sending a req_update for " << fname << "!" << std::endl;
                    if(write_x_bytes(sock_conn_id, sizeof(cloud_hdr_t) + fname_len + out_hdr.cloud_datalen, (uint8_t*)out_buf) < 0){
                      std::cout<<"Error while sending transmission!"<<std::endl;
                      return -1;
                    }
                  }
                  else{
                    //request patch!
                    struct cloud_hdr out_hdr;
                    out_hdr.cloud_reqtype = req_reqpatch;
                    out_hdr.cloud_userid = 0;
                    out_hdr.cloud_oldcrc = newest_crcs.at(fname);
                    out_hdr.cloud_newcrc = crc_in;
                    out_hdr.cloud_datalen = 0;
                    out_hdr.cloud_fnamelen = fname_len;

                    uint8_t* out_buf = new uint8_t[sizeof(cloud_hdr_t) + fname_len];
                    memcpy(out_buf, (uint8_t*) &out_hdr, sizeof(cloud_hdr_t));
                    memcpy(out_buf + sizeof(cloud_hdr_t), fname.c_str(), fname_len);

	                time_now = time(NULL);
                    time_tmp = localtime(&time_now);
                    std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Sending a req_reqpatch for " << fname << "!" << std::endl;
                    if(write_x_bytes(sock_conn_id, sizeof(cloud_hdr_t) + fname_len, out_buf) < 0){
                      std::cout<<"Error while sending transmission!"<<std::endl;
                      return -1;
                    }
                  }
                }
              }
            }


            //so we've dealt with each entry in message, lets see if we have more files
            for(auto itr = newest_crcs.begin(); itr != newest_crcs.end(); itr++){
              //if we have a crc for a file not contained in the message
              if(files_in.count((*itr).first) == 0){
                //send an updatefull!
                //send file!
                struct cloud_hdr out_hdr;
                out_hdr.cloud_reqtype = req_updatefull;
                out_hdr.cloud_userid = 0;
                out_hdr.cloud_oldcrc = 0;
                out_hdr.cloud_newcrc = (*itr).second;
                out_hdr.cloud_fnamelen = (*itr).first.size();


                std::ifstream file_in(path_to_file((*itr).first).c_str(), std::ios::binary);
                if(!file_in.is_open()){
                  std::cout<<"Error accessing file!"<<std::endl;
                  continue;
                }

                //find the length of the patch
                file_in.seekg (0, file_in.end);
                out_hdr.cloud_datalen = file_in.tellg();
                file_in.seekg (0, file_in.beg);

                std::cout << "data length: " << out_hdr.cloud_datalen << std::endl;

                //now we know all the lengths and stuff, let's allocate and fill!
                char* out_buf = new char[sizeof(cloud_hdr_t) + out_hdr.cloud_fnamelen + out_hdr.cloud_datalen];
                memcpy( out_buf, (char*) &out_hdr, sizeof(cloud_hdr_t) );
                memcpy( out_buf + sizeof(cloud_hdr_t), (*itr).first.c_str(), out_hdr.cloud_fnamelen );
                file_in.read(out_buf + sizeof(cloud_hdr_t) + out_hdr.cloud_fnamelen, out_hdr.cloud_datalen);
                file_in.close();

	            time_now = time(NULL);
                time_tmp = localtime(&time_now);
                std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Sending a req_updatefull for " << (*itr).first << "!" << std::endl;
                if(write_x_bytes(sock_conn_id, sizeof(cloud_hdr_t) + out_hdr.cloud_fnamelen + out_hdr.cloud_datalen, (uint8_t*)out_buf) < 0){
                  std::cout<<"Error while sending transmission!"<<std::endl;
                  return -1;
                }
              }
            }

            //send a req_notify that we're done replying to the reqall
            struct cloud_hdr out_hdr;
            out_hdr.cloud_reqtype = req_notify;
            out_hdr.cloud_userid = 0;
            out_hdr.cloud_oldcrc = 0;
            out_hdr.cloud_newcrc = 0;
            out_hdr.cloud_fnamelen = 0;
            out_hdr.cloud_datalen = 0;
            char* out_buf = new char[sizeof(cloud_hdr_t)];
            memcpy( out_buf, (char*) &out_hdr, sizeof(cloud_hdr_t) );
            
	        time_now = time(NULL);
            time_tmp = localtime(&time_now);
            std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Sending a req_notify!" << std::endl;
            if(write_x_bytes(sock_conn_id, sizeof(cloud_hdr_t) + out_hdr.cloud_fnamelen, (uint8_t*)out_buf) < 0){
              std::cout<<"Error while sending transmission!"<<std::endl;
              return -1;
            }

            break;
          }


        case req_delete:
          {
            //let's delete the file!
            char* to_delete = new char[cloud_in.cloud_fnamelen];
            if(read_x_bytes(sock_conn_id, cloud_in.cloud_fnamelen, (uint8_t*) to_delete) < 0){
              std::cout<<"Strange error while reading transmission!"<<std::endl;
              continue;
            }
            std::string del_file(to_delete, cloud_in.cloud_fnamelen);

            //remove file from our list of crc's
            newest_crcs.erase(del_file);
            //remove folder containing file/patches
            DIR* file_dir;

            std::string file_dir_string(cwd);
            file_dir_string += "/"+del_file;
            file_dir = opendir(file_dir_string.c_str());


            if(file_dir == nullptr){
              std::cout<<"Error while deleting directory!"<<std::endl;
              return -1;
            }
            //delete everything in the directory
            //move to directory
            chdir(file_dir_string.c_str());

	    struct dirent* this_dirent;
            while((this_dirent = readdir(file_dir))){
              if(this_dirent->d_type == DT_REG){
              	remove(this_dirent->d_name);
	      }
            }
            //move back down directories
            chdir(cwd.c_str());
            //remove folder
            remove(del_file.c_str());
            //cleanup
            delete[] to_delete;
            break;
          }
        case req_uptodate:
          //coolstory.jpg
          break;
        default:
          std::cout<<"Got an unknown reqtype! Ignoring!"<<std::endl;
          continue;
      }

      //we dealt with the message! let's update our crc's file now.
	  time_now = time(NULL);
      time_tmp = localtime(&time_now);
      std::cout << "(" << time_tmp->tm_hour << ":" << time_tmp->tm_min << ":" <<time_tmp->tm_sec << ") Updating .crcfile with..." << std::flush;
      std::ofstream crc_out(cwd+"/.crcfile", std::ios::out | std::ios::trunc);
      if(crc_out.is_open()){
        for(auto itr = newest_crcs.begin(); itr != newest_crcs.end(); itr++){
          std::cout<<(*itr).first<<".."<<std::flush;
          crc_out<<(*itr).first<<"|"<<(*itr).second<<"|";
        }
        crc_out.close();
      }
      else{
        std::cout<<"Error! Could not open .crcfile!"<<std::endl;
      }
      std::cout << "done!" << std::endl;

      //update timeout check
      time_rec = time(NULL);
    }//we've reacted to the messages. maybe we need to check again to see if more follow (loop)

    //we've timed out! let's go back to listening for connections
    close(sock_conn_id);
  }

  //how'd we get here?
  //let's stop listening now
  close(sock_id);

  //all done
  return 0;
}
