using namespace std;
#include <iostream>
#include <fstream>
#include <cstring>
#include <string>
#include <set>
#include <algorithm>
#include <iterator>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <zlib.h>
#include <cstdlib>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "cloud_protocol.h"
#include "cloud_utils.h"

#ifndef _HOME_
#define _HOME_ "/home/"
#endif

static const string SERVER_IP("127.0.0.1");

static const string ROOT_DRIVE = _HOME_;
static const string MAIN_SYNC_DRIVE = ROOT_DRIVE + "sync/";
static const string REPL_SYNC_DRIVE = ROOT_DRIVE + ".tmp/synced/";
static const string DIFF_SYNC_DRIVE = ROOT_DRIVE + ".tmp/diffs/";
static const int SLEEP_TIME = 5;
static const int SYNC_ALL_DELAY = 5;

/* socket descriptor for connection to server */
static int server_sd;

static string diff_file(string file)
{
    return DIFF_SYNC_DRIVE + file + string(".diff"); 
}

static string main_file(string f)
{
    return MAIN_SYNC_DRIVE + f;
}

static string repl_file(string f)
{
    return REPL_SYNC_DRIVE + f;
}

static int remove_file(string file)
{
    string cmd;
    cmd += "rm ";
    cmd += file;
    return system(cmd.c_str());
}

static int copy_file(string file, string output)
{
    string cmd;
    cmd += "cp ";
    cmd += file;
    cmd += " ";
    cmd += output;
    return system(cmd.c_str());
}

static int diff_files(string file1, string file2, string output) {
    string cmd;
    cmd += "diff -u --label=curr --label=curr ";
    cmd += file1; 
    cmd += " ";
    cmd += file2;
    cmd += " > ";
    cmd += output; 
    cout << "Performing the following command: " << cmd << endl;
    return system(cmd.c_str());
}

static int patch_file(string file, string patch) {
    string cmd;
    cmd += "patch -t ";
    cmd += file; 
    cmd += " < ";
    cmd += patch;
    cout << "Performing the following command: " << cmd << endl;
    return system(cmd.c_str());
}

static unsigned long int crc_file(string file) {
    ifstream f(file.c_str(),ios::in|ios::binary|ios::ate);
    unsigned int len = f.tellg();
    char* buf = new char[len];
    f.seekg(0,ios::beg);
    f.read(buf,len);
    f.close();
    uLong crc = crc32(0L, (unsigned char*)buf, len);
    delete []buf;
    return crc;
}

static int copy_repl2main(string f)
{
    return copy_file(repl_file(f), main_file(f));
}

static int copy_main2repl(string f)
{
    return copy_file(main_file(f), repl_file(f));
}

static int patch_main2repl(string f)
{
    return patch_file(repl_file(f), diff_file(f));
}

static int diff_main2repl(string f)
{
    return diff_files(repl_file(f), main_file(f), diff_file(f));
}

static set<string> list_files(DIR *dir)
{
    set<string> files;
    struct dirent *e;
    while ((e = readdir(dir)))
        if (e->d_type == DT_REG) 
            files.insert(e->d_name);
    return files;
}

static void printlist(set<string> s) 
{
    for (set<string>::const_iterator i = s.begin(), e = s.end(); i != e; i++) {
        cout << *i << "\n";
    }
    close(server_sd);
}


bool connected = false;

static void connect_to_server() {
    /* get a socket descriptor */
    if ((server_sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        cout << "Could not open a socket." << endl;
        exit(-1);
    }

    /* set up the server address */
    struct sockaddr_in server;
    server.sin_family = AF_INET;
    server.sin_port = htons(SERVER_PORT);
    server.sin_addr.s_addr = inet_addr(SERVER_IP.c_str());

    /* connect to server */
    cout << "Connecting to " << SERVER_IP << " on port " << SERVER_PORT << "..." << endl;
    while(connect(server_sd, (struct sockaddr*)&server, sizeof(server)) < 0) {
        cout << "Retrying in 5 seconds..." << endl; 
        sleep(5);
    }
    cout << "Connected!" << endl;
    connected = true;
}

static void close_connection() {
    close(server_sd);
    connected = false;
}

static void send_cloud_msg(cloud_hdr_t cloud_hdr, const char* filename, char* data) {
    if (!connected)
        connect_to_server();
    write_x_bytes(server_sd, sizeof(cloud_hdr_t), (uint8_t*)&cloud_hdr);
    write_x_bytes(server_sd, cloud_hdr.cloud_fnamelen, (uint8_t*)filename);
    if (cloud_hdr.cloud_datalen > 0)
        write_x_bytes(server_sd, cloud_hdr.cloud_datalen, (uint8_t*)data);
}

static void add_repl_files(set<string> s) 
{
    if (!s.empty()) {
        for (set<string>::const_iterator i = s.begin(), e = s.end(); i != e; i++) {
            cout << "Adding file " << *i << endl;
            /* update full file */
            if(copy_main2repl(*i) == 0) 
                cout << "Successfully added file " << *i << endl;
            ifstream repl(repl_file(*i).c_str(), ios::in|ios::binary|ios::ate);
            unsigned int repl_len = repl.tellg();
            char* repl_buf = new char[repl_len];
            repl.seekg(0,ios::beg);
            repl.read(repl_buf,repl_len);
            repl.close();
            cloud_hdr_t msg;
            msg.cloud_reqtype = req_updatefull;
            msg.cloud_fnamelen = (*i).length();
            msg.cloud_datalen = repl_len;
            msg.cloud_newcrc = crc_file(repl_file(*i));
            send_cloud_msg(msg,(*i).c_str(), repl_buf);
            delete []repl_buf;
        }
        close_connection();
    }
    else
        cout << "No files to add." << endl;
}

static void delete_repl_files(set<string> s) 
{
    if (!s.empty()) {
        for (set<string>::const_iterator i = s.begin(), e = s.end(); i != e; i++) {
            cout << "Removing file " << *i << endl;
            /* delete file from server */
            if (remove_file(repl_file(*i)) == 0) 
                cout << "Successfully removed file " << *i << endl;
            cloud_hdr_t msg;
            msg.cloud_reqtype = req_delete;
            msg.cloud_fnamelen = (*i).length();
            msg.cloud_datalen = 0;
            send_cloud_msg(msg,(*i).c_str(), NULL);
        }
        close_connection();
    }
    else
        cout << "No files to delete." << endl;
}

static void sync_repl_files(set<string> s) {
    if (!s.empty()) {
        for(set<string>::const_iterator i = s.begin(), e = s.end(); i != e; i++) {
            cout << "Syncing file " << *i << endl;
            unsigned long crc1 = crc_file(main_file(*i));
            unsigned long crc2 = crc_file(repl_file(*i));
            if (crc1 != crc2) {
                if (diff_main2repl(*i) == 0)
                    cout << "Successfully created diff file " << diff_file(*i) << endl;
                if (patch_main2repl(*i) == 0)
                    cout << "Successfully patched file " << *i << endl;
                /* send diff file */
                ifstream diff(diff_file(*i).c_str(), ios::in|ios::binary|ios::ate);
                unsigned int diff_len = diff.tellg();
                char* diff_buf = new char[diff_len];
                diff.seekg(0,ios::beg);
                diff.read(diff_buf,diff_len);
                diff.close();
                cloud_hdr_t msg;
                msg.cloud_reqtype = req_update;
                msg.cloud_fnamelen = (*i).length();
                msg.cloud_datalen = diff_len;
                msg.cloud_oldcrc = crc2;
                msg.cloud_newcrc = crc1;
                send_cloud_msg(msg, (*i).c_str(), diff_buf);
            } else {
                cout << "CRC matched...skipping" << endl;
            }
        }
        close_connection();
    }
    else 
        cout << "No files to sync." << endl;
}

static int check_all(set<string> files)
{
    cout << "Performing a check all." << endl;

    /* connect to server */
    connect_to_server();

    /* copy each file to replicator drive */
    for (set<string>::const_iterator i = files.begin(), e = files.end(); i != e; i++) 
        copy_main2repl(*i);

    /* allocate message*/
    unsigned int len = sizeof(cloud_hdr_t);
    for (set<string>::const_iterator i = files.begin(), e = files.end(); i != e; i++) 
        len += sizeof(uint16_t) + sizeof(uint64_t) + (*i).length();
    uint8_t* msg = (uint8_t*)malloc(len);

    /* fill in message */
    ((cloud_hdr_t*)msg)->cloud_reqtype = req_checkall;
    ((cloud_hdr_t*)msg)->cloud_datalen = len - sizeof(cloud_hdr_t);
    uint8_t* ptr = msg + sizeof(cloud_hdr_t);
    for (set<string>::const_iterator i = files.begin(), e = files.end(); i != e; i++) {
        *((uint16_t*)ptr) = ((*i).length());
        ptr += sizeof(uint16_t);
        *((uint64_t*)ptr) = (crc_file(main_file(*i)));
        ptr += sizeof(uint64_t);
        memcpy(ptr,(*i).c_str(),(*i).length());
        ptr += (*i).length();
    }

    /* send to server */
    if (write_x_bytes(server_sd, len, msg) <= 0) {
        cout << "Error while sending to server." << endl; 
        return -1;
    }

    /* wait for responses from server */
    while (1) {
        cout << "Waiting for responses from server." << endl;
        /* read the header */
        cloud_hdr_t hdr_buf;
        if (read_x_bytes(server_sd, sizeof(cloud_hdr_t), (uint8_t*)&hdr_buf) <= 0) {
            cout << "Error while reading from server." << endl;
            return -1;
        }
        cout << "Request type: " << (int)hdr_buf.cloud_reqtype << endl;
        /* if it's a notify, we're done, break */
        if (hdr_buf.cloud_reqtype == req_notify) { 
            cout << "Server finished responding." << endl;
            break;
        } 
        /* read the filename */
        char fname_buf[hdr_buf.cloud_fnamelen+1];
        fname_buf[hdr_buf.cloud_fnamelen] = 0;
        read_x_bytes(server_sd, hdr_buf.cloud_fnamelen, (uint8_t*)fname_buf);
        cout << "File name: " << fname_buf << endl;
        /* read the data */
        cout << "Data length: " << (int)hdr_buf.cloud_datalen << endl;
        char data_buf[hdr_buf.cloud_datalen+1];
        data_buf[hdr_buf.cloud_datalen] = 0;
        read_x_bytes(server_sd, hdr_buf.cloud_datalen, (uint8_t*)data_buf);
        /* process request */
        if (hdr_buf.cloud_reqtype == req_update) {
            cout << "Received a patch file." << endl;
            ofstream patch(diff_file(fname_buf).c_str(), ios::trunc); 
            patch << data_buf;
            patch.close();
            cout << data_buf << endl;
            patch_file(repl_file(fname_buf), diff_file(fname_buf));
            if (crc_file(main_file(fname_buf)) == hdr_buf.cloud_oldcrc)
                patch_file(main_file(fname_buf), diff_file(fname_buf));
        } else if (hdr_buf.cloud_reqtype == req_updatefull) {
            cout << "Received an update full." << endl;
            ofstream repl(repl_file(fname_buf).c_str(), ios::trunc);
            repl << data_buf;
            repl.close();
            copy_repl2main(fname_buf);
        } else if (hdr_buf.cloud_reqtype == req_reqpatch || hdr_buf.cloud_reqtype == req_reqfull) {
            cout << "Received an reqpatch or reqfull." << endl;
            ifstream repl(repl_file(fname_buf).c_str(), ios::in|ios::binary|ios::ate);
            unsigned int repl_len = repl.tellg();
            char* repl_buf = new char[repl_len];
            repl.seekg(0,ios::beg);
            repl.read(repl_buf,repl_len);
            repl.close();
            cloud_hdr_t reply;
            reply.cloud_reqtype = req_updatefull;
            reply.cloud_fnamelen = hdr_buf.cloud_fnamelen;
            reply.cloud_datalen = repl_len;
            reply.cloud_newcrc = crc_file(repl_file(fname_buf));
            write_x_bytes(server_sd, sizeof(cloud_hdr_t), (uint8_t*)&reply);
            write_x_bytes(server_sd, hdr_buf.cloud_fnamelen, (uint8_t*)fname_buf);
            write_x_bytes(server_sd, repl_len, (uint8_t*)repl_buf);
            delete []repl_buf;
        } else if (hdr_buf.cloud_reqtype == req_uptodate) {
            cout << "Received uptodate" << endl;
            continue; /* nothing to do here */
        } 
    }
    close_connection();
    cout << "Check all completed." << endl;
    return 0;
}

int main()
{
    /* main loop scans sync drive and updates as required */
    int time_to_check_all = 0;
    while (1) {
        cout << "------------------------------------------" << endl;
        cout << "Syncing (please do not exit right now)..." << endl;

        /* directory for main files */
        DIR *main_drive = opendir(MAIN_SYNC_DRIVE.c_str());
        if (!main_drive) {
            cerr << "Error opening sync drive." << endl;
            return 1;
        }

        /* directory for replicated files */
        DIR *repl_drive = opendir(REPL_SYNC_DRIVE.c_str());
        if (!repl_drive) {
            cerr << "Error opening replicator drive." << endl;
            return 1;
        }

        /* get set of files in main directory */
        set<string> main_files = list_files(main_drive);

        /* check all with the server on start up */
        if (time_to_check_all == 0) {
            check_all(main_files);
            time_to_check_all = SYNC_ALL_DELAY;
        }

        /* get set of files in replicator directory */
        set<string> repl_files = list_files(repl_drive);

        /* get set of new files in main directory */
        set<string> new_files;
        set_difference(main_files.begin(), main_files.end(), 
                repl_files.begin(), repl_files.end(), inserter(new_files, new_files.begin()));

        /* get set of deleted files in main directory */
        set<string> deleted_files;
        set_difference(repl_files.begin(), repl_files.end(), 
                main_files.begin(), main_files.end(), inserter(deleted_files, deleted_files.begin()));

        /* get set of synced files in main directory */
        set<string> existing_files;
        set_intersection(main_files.begin(), main_files.end(), 
                repl_files.begin(), repl_files.end(), inserter(existing_files, existing_files.begin()));

        /* add files to replicator drive */
        cout << "new files: \n";
        printlist(new_files);
        add_repl_files(new_files);

        /* delete files from replicator drive */
        cout << "deleted files: \n";
        printlist(deleted_files);
        delete_repl_files(deleted_files);

        /* sync files from replicator drive */
        cout << "existing files: \n";
        printlist(existing_files);
        sync_repl_files(existing_files);

        /* close the directories */
        closedir(main_drive);
        closedir(repl_drive);

        /* sleep until we have to do it again */
        cout << "Finished syncing. It is safe to exit now." << endl;
        cout << "------------------------------------------" << endl;
        cout << "Syncing again in:" << endl;
        for (int i = SLEEP_TIME; i > 0; i--) {
            cout << i << endl;
            sleep(1);
        }

        time_to_check_all--;
    }
}
