/*

PreQuel. Copyright Horst Samulowitz and Jessica Davies and Fahiem Bacchus 2006

 A QBF preprocessor based on performing extensive reasoning with the binary
 clause subtheory. 

Version 1.0

The software available at this site is copyright (c) 2006 by Horst Samulowitz 
Jessica Davies and Fahiem Bacchus. All right are reserved. 
Use of this software is permitted for non-commercial research purposes, and it may be copied only for that use. 
All copies must include this copyright message. This software and
any documentation and/or information supplied with it is distributed
on an as is basis. Horst Samulowitz and Jessica Davies and Fahiem Bacchus and the University of Toronto make no warranties, express or implied, including but not limited to implied warranties of merchantability and fitness for a particular purpose, regarding the documentation, functions or performance of such
software, documentation and/or information.

@misc{Samulowitz:Davies:Bacchus:2clsQ,
	author  = { Horst Samulowitz and Jessica Davies and Fahiem Bacchus },
	year    = { 2006 },
	title   = { A QBF preprocessor employing extensive binary clause reasoning },
	note    = { Available from
                    http://www.cs.toronto.edu/\~{}fbacchus/sat.html }
}

*/

//
// FILE: preprocessor.h
//

// OVERVIEW:
//
// This file contains the definitions of all the classes used in the preprocessor
// engine.  It contains numerous inline functions to access fields in the classes,
// used for information hiding purposes.  It also has the short inline functions
// used with the UPstack for simultaneous unit propagation of the binary and nary
// theory.

// There are four classes in this file.  The first is the preLit class which
// is the data structure used to represent literals in the preprocessor.  The
// second is the preNcls class used to represent nary clauses.  The third is
// the BCLSSTACKS class which is a stack used to hold binary clauses before
// they are added to the theory.  Finally, the EQVLITSTACKS class is a stack
// for holding all literals that are equivalent to another literal.  The stack
// is used by the support functions in preprocessor.cpp to remove these
// redundant literals from the theory.

//                               NEW FEATURES
//
// 1. This preprocessor version prints out a cnf file of the reduced theory.
// 2. A implication graph hashtable for speeding up the process of checking
//    if a binary clause exists has been added.
// 3. Function for Krom and simple subsumption were added.
// 4. Usage of preNcls field called active which is a flag for whether the
//    clause has been satisfied was made consistent.  This mean that when
//    a clause is set inActive, it is removed from the preLit field LitNcls
//    which is an array of the index the literal participated in.  Thus,
//    when iterating throught the LitNcls array, it is not required that
//    the activity of the nary clauses be checked because all of them
//    will be active.  A check for this was added to the checkReducedTheory()
//    function in checkpreproc.cpp

#ifndef __preprocessor_H
#define __preprocessor_H

#include <stdio.h>
#include "2clsEq.h"
#include "uniqueset.h"
#include "utilities.h"

// Flags used to run optional code.  See README file for descriptions.
//#define _DEBUG
//#define _EQVDEBUG
//#define _DETAILS
#define _STATS
//#define _CHECK

// The intial size of the literal array in a preNcls object.
#define INITNclsSize 5


// BUCKET STUFF
//#define initFreeBucketsChunkSize 10000
//#define ChunkSizeIncreaseFactor 4
//#define freeBucketsArraySize 100
//#define bclsTableSize 256
// END BUCKET STUFF

// Enumeration used in the depth-first searches of hyper-resolution.
enum colour {WHITE, GREY, BLACK};

// $$$ NEW $$$

enum UNDOtype {SETUNDEF, INCRLEN, SETACTIVE, OFFSTACK};
//enum UNDOtype {SETUNDEF, INCRLEN, SETACTIVE};
class UndoNode;
extern UndoNode * UndoStack;
extern int topUndoStack;
extern int sizeUndoStack;

// $$$ END NEW $$$



class preLit;
class preNcls;
class BCLSSTACKS;
class EQVLITSTACKS;

// These are the main class objects used in the preprocessor.
extern preLit * PRELITS;
extern preNcls * PRENCLS;
extern BCLSSTACKS * newBclsStack;
extern EQVLITSTACKS * equivLitStack;

#ifdef _STATS
// The variables record statistics about the preprocessor's
// performance.
extern int numValuedLits;
extern int numReducedNary;
extern int numSatCls;
extern int numSatNaryCls;
extern int numSatBcls;
extern int numAddBcls;
extern int numEquivLits;
extern int CURNLITS;
extern double start_time;
extern double end_time;
#endif // _STATS

// The number of nary clauses in the theory.
extern int NUMNCLS;

// The number of binary clauses in the theory.
extern int NUMBCLS;

// This boolean flag is true if the preprocessor found a satisfying
// truth assignment for the theory.
extern bool SATISFIED;

// This array contains a mapping to the original variable numbering
// scheme of used in the input cnf file.
extern int * orig_varnum;

// If the formula is found to be satisfiable, this boolean array is filled
// with a satisfying assignment of variables according to the internal
// variable numbering.
extern bool * var_assign;

// Number of variables remaining in the reduced theory.  Variables that
// were not assigned but nevertheless are not in the remaining clauses
// are not counted.
extern int NEW_NLITS;

// The following variables are used in a experimental method
// of reducing the copy time.
extern int * AccessedNclsList;
extern int sizeAccessedNclsList;
extern int * AccessedLitsList;
extern int sizeAccessedLitsList;

// BUCKET STUFF
//class bclsBucket;
//extern bclsBucket * nextFreeBucket;
//extern bclsBucket ** freeBuckets;
//extern int numFreeBuckets;
//extern int freeBucketsArrayIndex;
//extern int freeBucketsChunkSize;
// END BUCKET STUFF

// The following functions are define in preprocessor.cpp and used
// elsewhere.

// The main preprocessor function.
void preprocessor();

// Adds a literal to the equivLitStack.
void pushequivLitStack(int L);

// Adds a binary clause, actually each implication independently, to the
// theory.
bool addBcls(int L, int impliedL);

// Functions used for finding strongly connected components.
void initSCC();
void findSCC();
void incrFindSCC();

// Jessica: performs universal reduction on a clause
bool universalReduction(preNcls *Cls, int clsIndex);

// These print out parts of the data structures.
void printDS();
void printReducedTheory();
void fprintReducedTheory();

#ifdef _STATS
// These print out many of the preprocessor statistics.
void printPreprocStats();
void printReport();
#endif // _STATS

// This function creates a cnf file containing the reduced theory generated
// by the preprocessor.
void cnfoutput();

void cnfoutputOrigTheory();

// This function appends a line to the file preprocessor.dat with a
// summary of the results of the preprocessor.
void recordSummary();

// This function prints out a brief report of the preprocessor's results.
void printSummary();

// This class represents the literals as they are used during the preprocessor.
class preLit {
 public:
  // The next variable is the integer representation of the literal.
  int num;

  // The literals value.
  tval val;

  // The literal that replaced this literal in the theory.
  int equivLit;

  // The is integer array contains the literals implied by this literal
  // and hence a representation of the binary clauses.
  // For example, if the binary clause (1 v -2) was part of the theory,
  // then the two implications representing the clause are -1->-2 and
  // 2->1.  Thus, -2 will be stored in -1's LitBcls array and 1 will be
  // stored on 2's LitBcls array.
  int * LitBcls;

  // The actual size of the LitBcls array.
  int maxBcls;

  // The current number of literals in LitBcls.
  int numBcls;

  // This integer array contains the indices of the nary clauses that the
  // literals appears in.  The indices are the indices in the PRENCLS
  // data structure.
  int * LitNcls;

  // The actual size of the LitNcls array.
  int maxNcls;

  // The current number of nary clause indices this literal is in.
  int numNcls;

  // A visited flag used by the SCC algorithm and the binary resolution
  // algorithm.
  bool visited;

  // This field is used by the SCC algorithm in findSCC.cpp to keep track
  // of the order the literals are visited in.  It is also used to count
  // the number of times literals are visited during binary resolution
  // for testing that algorithms efficiency.
  int visitOrder;

  // This field is used by the SCC algorithm in findSCC.cpp to mark literals
  // that have been assigned a component.  It is also used as a visited flag
  // by helper functions in the hyper-resolution algorithm.
  bool inComponent;

  // Another flag used by the SCC algoirthm.
  bool onStack;

  // This variable indicates whether a new binary clause was recently added to
  // this literal.  If so, it must be visited by the incrFindSCC function that
  // finds new Strongly Connected Components without checking every literal.
  bool bclsAdded;

  // Flags used with naryUPstack and bclsUPstack to indicated whether literals
  // have been progated through the naryUPstack and blcsUPstack.
  bool naryProp;
  bool bclsProp;


  //REMOVE
  // Temporary addition for second version of Binary Resolution DFS
  UniqueSet * implications;
  int * implList;
  int maxImpl;
  int numImpl;
  //REMOVE

  // This variable is used by the depth-first searches in Hyper-resolution
  // to distinguish between unprocessed nodes, those in the middle of being
  // processed, and those finished being processed.
  colour state;

  // This variable is used by HypRes to temporarily store the value of the
  // literal while computing new implications during Hyper-resolution.
  tval HR_val;

  // This flag indicates whether this literal has been accessed by the
  // current invocation of HypRes while computing new implications.
  // If a literal is accessed for the first time, the actual value from
  // "val" must be copied to "HR_val".  If it has already been accessed,
  // no copying is needed.  After each HypRes invocation, the flag is
  // reset to false.
  bool accessed;

  // Horst: Added Prefixlevel and Type
  int  nPrefixLevel;
  bool bExistential;
  // End

  // Jessica: stores the lowest prefix level literal in the scc 
  // whose root is this literal
  int  lowestEquiv;

  // BUCKET STUFF
  //bclsBucket * bclsTable;
  // END BUCKET STUFF
};

// Converts a literal's number to a pointer to its location in the
// PRELITS data structure.
inline preLit * intToPreLit(int lit) {
  return(PRELITS + lit);
}

// The set of functions following provide a layer of abstraction around
// the preLit class so that the internal configuration of the class
// can be changed without affecting the code that uses it.  Each field
// has four corresponding functions that either return the field value
// or set the field to a new value.  Two functions take the literals
// integral number and two take a pointer as arguments.
inline int getNum(preLit * L) {
  return(L->num);
}

inline void setNum(preLit * L, int num) {
  L->num = num;
}

inline void setNum(int lit, int num) {
  PRELITS[lit].num = num;
}

inline tval getVal(preLit * L) {
  return(L->val);
}

inline tval getVal(int lit) {
  return(PRELITS[lit].val);
}

inline void setVal(preLit * L, tval val) {
  L->val = val;
}

inline void setVal(int lit, tval val) {
  PRELITS[lit].val = val;
}

inline int getEquivLit(preLit * L) {
  return(L->equivLit);
}

inline int getEquivLit(int lit) {
  return(PRELITS[lit].equivLit);
}

inline int getEquivLitCorrectly(int lit) {
  while(lit !=  getEquivLit(lit)) {
   lit = getEquivLit(lit);
  }
  return(lit);
}


inline void setEquivLit(preLit * L, int equivLit) {
  L->equivLit = equivLit;
}

inline void setEquivLit(int lit, int equivLit) {
  PRELITS[lit].equivLit = equivLit;
}

inline int * getLitBcls(preLit * L) {
  return(L->LitBcls);
}

inline int * getLitBcls(int lit) {
  return(PRELITS[lit].LitBcls);
}

inline void setLitBcls(preLit * L, int * LitBcls) {
  L->LitBcls = LitBcls;
}

inline void setLitBcls(int lit, int * LitBcls) {
  PRELITS[lit].LitBcls = LitBcls;
}

inline int getLitBclsElem(preLit * L, int index) {
  return(L->LitBcls[index]);
}

inline int getLitBclsElem(int lit, int index) {
  return(PRELITS[lit].LitBcls[index]);
}

inline void setLitBclsElem(preLit * L, int index, int value) {
  L->LitBcls[index] = value;
}

inline void setLitBclsElem(int lit, int index, int value) {
  PRELITS[lit].LitBcls[index] = value;
}

inline int getMaxBcls(preLit * L) {
  return(L->maxBcls);
}

inline int getMaxBcls(int lit) {
  return(PRELITS[lit].maxBcls);
}

inline void setMaxBcls(preLit * L, int maxBcls) {
  L->maxBcls = maxBcls;
}

inline void setMaxBcls(int lit, int maxBcls) {
  PRELITS[lit].maxBcls = maxBcls;
}

inline int getNumBcls(preLit * L) {
  return(L->numBcls);
}

inline int getNumBcls(int lit) {
  return(PRELITS[lit].numBcls);
}

inline void setNumBcls(preLit * L, int numBcls) {
  L->numBcls = numBcls;
}

inline void setNumBcls(int lit, int numBcls) {
  PRELITS[lit].numBcls = numBcls;
}

inline int * getLitNcls(preLit * L) {
  return(L->LitNcls);
}

inline int * getLitNcls(int lit) {
  return(PRELITS[lit].LitNcls);
}

inline void setLitNcls(preLit * L, int * LitNcls) {
  L->LitNcls = LitNcls;
}

inline void setLitNcls(int lit, int * LitNcls) {
  PRELITS[lit].LitNcls = LitNcls;
}

inline int getLitNclsElem(preLit * L, int index) {
  return(L->LitNcls[index]);
}

inline int getLitNclsElem(int lit, int index) {
  return(PRELITS[lit].LitNcls[index]);
}

inline void setLitNclsElem(preLit * L, int index, int value) {
  L->LitNcls[index] = value;
}

inline void setLitNclsElem(int lit, int index, int value) {
  PRELITS[lit].LitNcls[index] = value;
}


// *********************************************************
// Horst: ADDED Prefix-Operators
// *********************************************************

inline void setLitPrefixLevel(preLit * L, int nLevel) {
  L->nPrefixLevel = nLevel;
}

inline void setLitPrefixLevel(int l, int nLevel) {
  PRELITS[l].nPrefixLevel = nLevel;
}

inline int getLitPrefixLevel(preLit* L) {
  return L->nPrefixLevel;
}

inline int getLitPrefixLevel(int l) {
  return (PRELITS[l].nPrefixLevel);
}

inline void setLitPrefixType(preLit* L, bool bExistential) {
  L->bExistential = bExistential;
}

inline void setLitPrefixType(int l, bool bExistential) {
  PRELITS[l].bExistential = bExistential;
}

inline bool isLitExistential(preLit* L) {
  return L->bExistential;
}

inline bool isLitExistential(int l) {
  return (PRELITS[l].bExistential);
}

// Jessica
inline int getLowestEquiv(preLit* L) {
  return L->lowestEquiv;
}

inline int getLowestEquiv(int l) {
  return (PRELITS[l].lowestEquiv);
}
inline void setLowestEquiv(preLit* L, int u) {
  L->lowestEquiv = u;
}

inline void setLowestEquiv(int l, int u) {
  PRELITS[l].lowestEquiv = u;
}

// *********************************************************
// END: ADDED Prefix-Operators
// *********************************************************




inline int getMaxNcls(preLit * L) {
  return(L->maxNcls);
}

inline int getMaxNcls(int lit) {
  return(PRELITS[lit].maxNcls);
}

inline void setMaxNcls(preLit * L, int maxNcls) {
  L->maxNcls = maxNcls;
}

inline void setMaxNcls(int lit, int maxNcls) {
  PRELITS[lit].maxNcls = maxNcls;
}

inline int getNumNcls(preLit * L) {
  return(L->numNcls);
}

inline int getNumNcls(int lit) {
  return(PRELITS[lit].numNcls);
}

inline void setNumNcls(preLit * L, int numNcls) {
  L->numNcls = numNcls;
}

inline void setNumNcls(int lit, int numNcls) {
  PRELITS[lit].numNcls = numNcls;
}

inline bool getVisited(preLit * L) {
  return(L->visited);
}

inline bool getVisited(int lit) {
  return(PRELITS[lit].visited);
}

inline void setVisited(preLit * L, bool visited) {
  L->visited = visited;
}

inline void setVisited(int lit, bool visited) {
  PRELITS[lit].visited = visited;
}

inline int getVisitOrder(preLit * L) {
  return(L->visitOrder);
}

inline int getVisitOrder(int lit) {
  return(PRELITS[lit].visitOrder);
}

inline void setVisitOrder(preLit * L, int visitOrder) {
  L->visitOrder = visitOrder;
}

inline void setVisitOrder(int lit, int visitOrder) {
  PRELITS[lit].visitOrder = visitOrder;
}

inline bool getInComponent(preLit * L) {
  return(L->inComponent);
}

inline bool getInComponent(int lit) {
  return(PRELITS[lit].inComponent);
}

inline void setInComponent(preLit * L, bool inComponent) {
  L->inComponent = inComponent;
}

inline void setInComponent(int lit, bool inComponent) {
  PRELITS[lit].inComponent = inComponent;
}

inline bool getOnStack(preLit * L) {
  return(L->onStack);
}

inline bool getOnStack(int lit) {
  return(PRELITS[lit].onStack);
}

inline void setOnStack(preLit * L, bool onStack) {
  L->onStack = onStack;
}

inline void setOnStack(int lit, bool onStack) {
  PRELITS[lit].onStack = onStack;
}

inline bool getBclsAdded(preLit * L) {
  return(L->bclsAdded);
}

inline bool getBclsAdded(int lit) {
  return(PRELITS[lit].bclsAdded);
}

inline void setBclsAdded(preLit * L, bool bclsAdded) {
  L->bclsAdded = bclsAdded;
}

inline void setBclsAdded(int lit, bool bclsAdded) {
  PRELITS[lit].bclsAdded = bclsAdded;
}

inline bool getNaryProp(preLit * L) {
  return(L->naryProp);
}

inline bool getNaryProp(int lit) {
  return(PRELITS[lit].naryProp);
}

inline void setNaryProp(preLit * L, bool naryProp) {
  L->naryProp = naryProp;
}

inline void setNaryProp(int lit, bool naryProp) {
  PRELITS[lit].naryProp = naryProp;
}

inline bool getBclsProp(preLit * L) {
  return(L->bclsProp);
}

inline bool getBclsProp(int lit) {
  return(PRELITS[lit].bclsProp);
}

inline void setBclsProp(preLit * L, bool bclsProp) {
  L->bclsProp = bclsProp;
}

inline void setBclsProp(int lit, bool bclsProp) {
  PRELITS[lit].bclsProp = bclsProp;
}

inline colour getState(preLit * L) {
  return(L->state);
}

inline colour getState(int lit) {
  return(PRELITS[lit].state);
}

inline void setState(preLit * L, colour state) {
  L->state = state;
}

inline void setState(int lit, colour state) {
  PRELITS[lit].state = state;
}

// See below for definitions.
inline void checkLitsAccessed(preLit * L);
inline void checkLitsAccessed(int lit);
inline void setLitsAccessed(preLit * L);
inline void setLitsAccessed(int lit);

// The wrapper functions for HR_val are different from the others
// because they must contain functionality to mark that this literal
// has been accessed and to copy the value stored in val when necessary.
inline tval getHR_val(preLit * L) {
  checkLitsAccessed(L);
  return(L->HR_val);
}

inline tval getHR_val(int lit) {
  checkLitsAccessed(lit);
  return(PRELITS[lit].HR_val);
}

inline void setHR_val(preLit * L, tval HR_val) {
  setLitsAccessed(L);
  L->HR_val = HR_val;
}

inline void setHR_val(int lit, tval HR_val) {
  setLitsAccessed(lit);
  PRELITS[lit].HR_val = HR_val;
}

inline void setHR_val_init(int lit, tval HR_val) {
  PRELITS[lit].HR_val = HR_val;
}

// Used when reseting the accessed variable.
inline void setLitAccessedFalse(int lit) {
  PRELITS[lit].accessed = false;
}

// Checks if this is the first time the literal is accessed in this
// invocation of HypRes and sets the accessed flag to true.
// Copies over the correct value from val.  Then adds the literals
// to the list of accessed literals.
inline void checkLitsAccessed(preLit * L) {
  if(!L->accessed) {
    L->accessed = true;
    setHR_val(L, getVal(L));
    AccessedLitsList[sizeAccessedLitsList++] = L->num;
  }
}

// Same as above using the literals integer value.
inline void checkLitsAccessed(int lit) {
  if(!PRELITS[lit].accessed) {
    PRELITS[lit].accessed = true;
    setHR_val(lit, getVal(lit));
    AccessedLitsList[sizeAccessedLitsList++] = lit;
  }
}

// The next to function just set the literals to accessed
// and add it to the list.
inline void setLitsAccessed(preLit * L) {
  if(!L->accessed) {
    L->accessed = true;
    AccessedLitsList[sizeAccessedLitsList++] = L->num;
  }
}

inline void setLitsAccessed(int lit) {
  if(!PRELITS[lit].accessed) {
    PRELITS[lit].accessed = true;
    AccessedLitsList[sizeAccessedLitsList++] = lit;
  }
}

// The following two function return true if a literals is still active.
// This means the literal has not been assigned a value or equivalent
// literal.
inline bool active(preLit * L) {
  return((L->num == L->equivLit) && (L->val == U));
}

inline bool active(int L) {
  return((L == getEquivLit(L)) && (getVal(L) == U));
}

// Converts internal literal numbering to external literal numbering for
// printing purposes.
inline int IntLitToExt(int L) {
  return((L + 2) / ((L % 2)? 2: -2));
}

// This class represents nary clauses in the processor.
class preNcls {
 public:

  // An array of the literals in the clause.
  int * lits;

  // The current number of active literals in the clause.
  int curLen;

  // True is the clause is not yet satisfied.
  bool active;

  // These variables are used by HypRes to temporarily store the length and
  // state of a clause while computing new implications during
  // Hyper-resolution.
  int HR_curLen;
  bool HR_active;

  // A flag indicating whether the current invocation of HypRes has accessed
  // this clause.
  bool accessed;
};

// Converts a clause's number to a pointer to its location in the
// PRENCLS data structure.
inline preNcls * intToPreNcls(int cls) {
  return(PRENCLS + cls);
}

// The set of functions following provide a layer of abstraction around
// the preNcls class so that the internal configuration of the class
// can be changed without affecting the code that uses it.  Each field
// has four corresponding functions that either return the field value
// or set the field to a new value.  Two functions take the literals
// integral number and two take a pointer as arguments.
inline int * getLits(preNcls * C) {
  return(C->lits);
}

inline int * getLits(int cls) {
  return(PRENCLS[cls].lits);
}

inline void setLits(preNcls * C, int * lits) {
  C->lits = lits;
}

inline void setLits(int cls, int * lits) {
  PRENCLS[cls].lits = lits;
}

inline int getLitsElem(preNcls * C, int index) {
  return(C->lits[index]);
}

inline int getLitsElem(int cls, int index) {
  return(PRENCLS[cls].lits[index]);
}

inline void setLitsElem(preNcls * C, int index, int value) {
  C->lits[index] = value;
}

inline void setLitsElem(int cls, int index, int value) {
  PRENCLS[cls].lits[index] = value;
}

inline int getCurLen(preNcls * C) {
  return(C->curLen);
}

inline int getCurLen(int cls) {
  return(PRENCLS[cls].curLen);
}

inline void setCurLen(preNcls * C, int curLen) {
  C->curLen = curLen;
}

inline void setCurLen(int cls, int curLen) {
  PRENCLS[cls].curLen = curLen;
}

inline bool getActive(preNcls * C) {
  return(C->active);
}

inline bool getActive(int cls) {
  return(PRENCLS[cls].active);
}

inline void setActive(preNcls * C, bool active) {
  C->active = active;
}

inline void setActive(int cls, bool active) {
  PRENCLS[cls].active = active;
}

// See below for definitions.
inline void checkNclsAccessed(preNcls * C);
inline void checkNclsAccessed(int cls);
inline void setNclsAccessed1(preNcls * C);
inline void setNclsAccessed1(int cls);
inline void setNclsAccessed2(preNcls * C);
inline void setNclsAccessed2(int cls);

// The wrapper functions for HR_curLen and HR_active are different from
// the others because they must contain functionality to mark that this
// clause has been accessed and to copy the value on curLen into HR_curLen
// and active into HR_active when these fields are accessed for the first
// time.

inline int getHR_curLen(preNcls * C) {
  checkNclsAccessed(C);
  return(C->HR_curLen);
}

inline int getHR_curLen(int cls) {
  //checkNclsAccessed(cls);
  return(PRENCLS[cls].HR_curLen);
}

inline void setHR_curLen(preNcls * C, int HR_curLen) {
  setNclsAccessed1(C);
  C->HR_curLen = HR_curLen;
}

inline void setHR_curLen(int cls, int HR_curLen) {
  //setNclsAccessed1(cls);
  PRENCLS[cls].HR_curLen = HR_curLen;
}


inline bool getHR_active(preNcls * C) {
  checkNclsAccessed(C);
  return(C->HR_active);
}

inline bool getHR_active(int cls) {
  checkNclsAccessed(cls);
  return(PRENCLS[cls].HR_active);
}

inline void setHR_active(preNcls * C, bool HR_active) {
  setNclsAccessed2(C);
  C->HR_active = HR_active;
}

inline void setHR_active(int cls, bool HR_active) {
  setNclsAccessed2(cls);
  PRENCLS[cls].HR_active = HR_active;
}


// Used when reseting the accessed variable.
inline void setNclsAccessedFalse(int cls) {
  PRENCLS[cls].accessed = false;
}

// When this clause is accessed by a HypRes invocation for the first time
// one of the next two functions is used to copy the values from the
// permanent curLen and active fields to the temporary ones.  The
// clause is also set to accessed and added to the accessed list.

inline void checkNclsAccessed(preNcls * C) {
  if(!C->accessed) {
    C->accessed = true;
    setHR_curLen(C, getCurLen(C));
    setHR_active(C, getActive(C));
    AccessedNclsList[sizeAccessedNclsList++] = C - PRENCLS;
  }
}

inline void checkNclsAccessed(int cls) {
  if(!PRENCLS[cls].accessed) {
    PRENCLS[cls].accessed = true;
    setHR_curLen(cls, getCurLen(cls));
    setHR_active(cls, getActive(cls));
    AccessedNclsList[sizeAccessedNclsList++] = cls;
  }
}

// The two functions are used with the setHR_curLen functions to set the
// clause to accessed and add it to the accessed clause list.  They also
// set up the active fields for the first time.
inline void setNclsAccessed1(preNcls * C) {
  if(!C->accessed) {
    C->accessed = true;
    setHR_active(C, getActive(C));
    AccessedNclsList[sizeAccessedNclsList++] = C - PRENCLS;
  }
}

inline void setNclsAccessed1(int cls) {
  if(!PRENCLS[cls].accessed) {
    PRENCLS[cls].accessed = true;
    setHR_active(cls, getActive(cls));
    AccessedNclsList[sizeAccessedNclsList++] = cls;
  }
}

// The two functions are used with the setHR_active functions to set the
// clause to accessed and add it to the accessed clause list.  They also
// set up the curLen fields for the first time.
inline void setNclsAccessed2(preNcls * C) {
  if(!C->accessed) {
    C->accessed = true;
    setHR_curLen(C, getCurLen(C));
    AccessedNclsList[sizeAccessedNclsList++] = C - PRENCLS;
  }
}

inline void setNclsAccessed2(int cls) {
  if(!PRENCLS[cls].accessed) {
    PRENCLS[cls].accessed = true;
    setHR_curLen(cls, getCurLen(cls));
    AccessedNclsList[sizeAccessedNclsList++] = cls;
  }
}

// Only one BCLSSTACKS instance is required.  It stores the binary clauses
// generated through simplification of the nary clauses in the theory.
// Binary clauses are stored as two sequencial integers in the clauses
// array which represent the two literals.  Thus, when to remove or add
// a binary clause to the stack, two literals must be popped or pushed.
//  Also, if size = n, then the stack can store n/2 binary clauses.
class BCLSSTACKS {
 public:

  // The array of literals in new binary clauses.
  int * clauses;

  // The size of the clauses array.
  int size;

  // The index of the top of the stack.
  int top;
};

// Assumes stack is not empty when called.
inline void popnewBclsStack(int &L1, int &L2) {
  L2 = newBclsStack->clauses[newBclsStack->top--];
  L1 = newBclsStack->clauses[newBclsStack->top--];
}

// Returns true if empty.
inline bool emptynewBclsStack() {
  return(newBclsStack->top == -1);
}

// This class represents a stack of all the literals that have just
// been made equivalent to another literal and must be removed from
// the binary and nary clause theory.
class EQVLITSTACKS {
 public:

  // The array of literals to be removed.
  int * lits;

  // The size of the lits array.
  int size;

  // The index of the temporary top of the stack.
  int top;

  // realTop is used to allow the stack to be run through twice, while processing
  // the equivalent literals, once to remove them from the binary clauses and
  // once for the nary clauses.
  int realTop;
};

// Assume stack is not empty when called.
inline int popequivLitStack() {
  return(equivLitStack->lits[equivLitStack->top--]);
}

// Returns true if empty.
inline bool emptyequivLitStack() {
  return(equivLitStack->top == -1);
}

// $$$ NEW $$$

class UndoNode {
 public:
  UNDOtype type;
  int index;
};


inline void checkSizeUndoStack() {
  if(topUndoStack == (sizeUndoStack - 1)) {
    //printf("@@@@@@@@@@@@@@ UndoStack needed size adjustment @@@@@@@@@@@@@@\n");
    sizeUndoStack *= 4;
    UndoNode * temp = new UndoNode[sizeUndoStack];
    if(!temp)
      panic("Memory Allocation Failure #27P\n");
    for(int i = 0; i <= topUndoStack; i++) {
	temp[i].type = UndoStack[i].type;
	temp[i].index = UndoStack[i].index;
    }
    delete [] UndoStack;
    UndoStack = temp;
  }
}

inline void setVal_Undo(int lit, tval val) {
  checkSizeUndoStack();
  UndoStack[++topUndoStack].type = SETUNDEF;
  UndoStack[topUndoStack].index = lit;
  PRELITS[lit].val = val;
}

inline void DecrCurLen_Undo(int cls) {
  checkSizeUndoStack();
  UndoStack[++topUndoStack].type = INCRLEN;
  UndoStack[topUndoStack].index = cls;
  PRENCLS[cls].curLen--;
}

inline void setInActive_Undo(int cls) {
  checkSizeUndoStack();
  UndoStack[++topUndoStack].type = SETACTIVE;
  UndoStack[topUndoStack].index = cls;
  PRENCLS[cls].active = false;
}

inline void setOnStack_Undo(int lit) {
  checkSizeUndoStack();
  UndoStack[++topUndoStack].type = OFFSTACK;
  UndoStack[topUndoStack].index = lit;
  PRELITS[lit].onStack = true;
}

inline void IncrCurLen(int cls) {
   PRENCLS[cls].curLen++;
}


// $$$ END NEW $$$

// These are the short inline functions used with the UPstack.  The longer
// functions are in uplit.cpp.
extern UniqueSet * UPstack;

void pushUPstack(int L);
void UnitPropLit(int L);

inline int popUPstack() {
  int L = UPstack->pop();
  int eqvL = getEquivLit(L);
  return(eqvL);
}

// Returns trye if empty.
inline bool emptyUPstack() {
  return(UPstack->size == 0);
}

// Unit propagates all literals in the UPstack.
inline void propagateUPstack() {
  while(!CONTRADICTION.val && !emptyUPstack())
    UnitPropLit(popUPstack());
  if(CONTRADICTION.val) {
    UPstack->empty();
  }
}

// BUCKET STUFF
//class bclsBucket {
//public:
//int lit;
//bclsBucket * next;
//};

//void initFreeBuckets();
//void allocMoreFreeBuckets();
//bclsBucket * getFreeBucket();
//void initBclsTable(int L);
//bool addBclsToTable(int L1, int L2);
//bool findBclsTable(int L1, int L2);
//bool deleteBclsFromTable(int L1, int L2);
//void printBclsTables();
//void printBclsTable(int lit);
// END BUCKET STUFF


#endif /* __preprocessor_H */
