/*

PreQuel. Copyright Horst Samulowitz and Jessica Davies and Fahiem Bacchus 2006

 A QBF preprocessor based on performing extensive reasoning with the binary
 clause subtheory. 

Version 1.0

The software available at this site is copyright (c) 2006 by Horst Samulowitz 
Jessica Davies and Fahiem Bacchus. All right are reserved. 
Use of this software is permitted for non-commercial research purposes, and it may be copied only for that use. 
All copies must include this copyright message. This software and
any documentation and/or information supplied with it is distributed
on an as is basis. Horst Samulowitz and Jessica Davies and Fahiem Bacchus and the University of Toronto make no warranties, express or implied, including but not limited to implied warranties of merchantability and fitness for a particular purpose, regarding the documentation, functions or performance of such
software, documentation and/or information.

@misc{Samulowitz:Davies:Bacchus:2clsQ,
	author  = { Horst Samulowitz and Jessica Davies and Fahiem Bacchus },
	year    = { 2006 },
	title   = { A QBF preprocessor employing extensive binary clause reasoning },
	note    = { Available from
                    http://www.cs.toronto.edu/\~{}fbacchus/sat.html }
}

*/

//
// FILE: preprocessor.cpp
//

// OVERVIEW:
//
// The file contains the main function for the preprocessor (preprocessor()),
// which contains the 16 main steps of the preprocessor.  The basic outline
// of the 16 steps is as follows.  First, the two main data structures,
// PRELITS and PRENCLS are initialized and the theory is copied into them
// from the lenNclauses data structure.  Then, all the unit clauses in the
// theory are unit propagated.  The strongly-connected components algorithm
// is then run.  All redundant literals are then removed from theory and
// replaced in clauses by their equivalent literals.  Optionally, binary
// resolution is run at this point.  Unit clauses produced by SCC and binary
// resolution are unit propagated.  Hyper-resolution is then run once over
// all literals.  The cycle of SCC algorithm, BinRes, unit propagation, and
// HypRes is repeated until the theory cannot be simplified further.  Then,
// the preprocessor terminated and prints some statistics.

// This file has three sections.  The first part contains a large number of
// support functions called by preprocessor() during its execution.  These
// function manipulate that data structures and perform operations such as
// unit propagation and binary resolution.  Some of the functions called by
// preprocessor() are not in this file.  One set of unit propagation
// function is found in uplit.cpp.  The SCC algorithm is in findscc.cpp.
// The hyper-resolution is in hypres.cpp.  Eventually, other function will
// probably be moved to different files for better organization.

// The second part is the actual preprocessor() function.  As mentioned
// earlier, it is basically a sequence of steps.

// The third sections is at the bottom of the file.  It contains function
// that print out all the different parts of the data structures.  Some
// of these functions print to files and other print to the screen.  There
// are also some functions that print reports of the progress of the
// preprocessor and summaries of its performance.

#include "preprocessor.h"

//REMOVE
#define _ONEUPSTACK
//#define _TWOUPSTACK
//REMOVE

// Optionally run the binary resolution before hyper-resolution.  I am
// uncertain if this make the program faster or slower, but initial
// tests indicate it is slower.
//#define _BINRES

// The initial size of the Bcls array in a preLit object.
#define INITBclsPerLit 10
// The initial size of the Ncls array in a preLit object.
#define INITNclsPerLit 100

//REMOVE
// The initial size of the implList used to temporarily store
// the transitive closure during BinResDFS3.
#define INITImplPerLit 100
//REMOVE

// The initial size of the new binary clause stack. This
// value is multiplied by the number of literals in the
// problem.
#define INITBclsStackSize 1

//#################### VARIABLE DECLARATIONS #######################//

// This is the name of the file containing the input boolean formula.
extern char * filename;

// This is the name of the output file in which the simplified theory
// is stored when the preprocessor is completed.
extern char * outfile;

// Declaration of the data structure for the literals.
preLit * PRELITS;

// Declaration of the data structure for the nary clauses.
preNcls * PRENCLS;

// A stack for unit propagation of the nary clauses independent of the
// binary clauses.
UniqueSet * naryUPstack;

// A stack for unit propagation of the binary clauses independent of the
// nary clauses.
UniqueSet * bclsUPstack;

// A stack for unit propagating the whole theory simultaneously.
extern UniqueSet * UPstack;

// A stack of binary clauses to be added to the theory.
BCLSSTACKS * newBclsStack;

// A stack of equivalent literals to be removed from the theory.
EQVLITSTACKS * equivLitStack;

// Elements of this array are marked true if the corresponding literal
// was found to be forced true by binary resolution.
bool * UPlist;

// This flag is set true if any unit propagations are found by BinRes.
bool foundNewUP;

// The number of nary clauses in the original theory.
int NUMNCLS;

// The number of binary clauses in the original theory.
int NUMBCLS = 0;

// This boolean flag is true if the preprocessor found a satisfying
// truth assignment for the theory.
bool SATISFIED = false;

//REMOVE
//temp - used with binary resolution DFS
UniqueSet * tempSet;
//REMOVE

// The following variables are used by Hyper-resolution.

// A stack of the variable to be temporarily unit propagated
// while hyper-resolving a literal.
extern int * UnitPropStack;
// This flag is used by runHypRes() to indicate whether
// hyper-resolution found any new binary clauses.  This flag helps
// determine when the preprocessor cannot simplify the theory
// anymore and should terminate.
extern bool foundNewBcls;
// This flag is used by runHypRes to indicated if it found any
// forced literals.  That is, whether it found any literals that
// when set true caused a contradiction, forcing the negation of
// the literal to be true.
extern bool foundNewForced;

#ifdef _STATS
// Used for to tell the unit propagation that its is being run from
// HypRes.  Used for tracing which literals are UP during HypRes.
bool inHypRes = false;

// Variables for statistics gathering.
int numValuedLits = 0;        // number of valued literals
int numReducedNary = 0;       // number of nary clauses reduced to binary
int numSatCls = 0;            // total number of satisfied clauses
int numSatNaryCls = 0;        // number of satisfied nary clauses
int numSatBcls = 0;           // number of satisfied binary clauses
int numAddBcls = 0;           // number of new binary clauses added to theory
int numEquivLits = 0;         // number of redundant literals found.
int numInitUnits = 0;         // number of initial unit clauses in the theory
int CURNLITS;                 // number of literals left in the theory
int deleteCount = 0;          // number of times delete was called
int DFScounter = 0;           // number of calls to BinResDFS
int iterNum;                  // number of cycles of SCC, HypRes, etc.
double start_time = 0.0;      // starting time of preprocessor
double end_time = 0.0;        // ending time of preprocessor
double dfs_start_time;        // used to time binary resolution
double dfs_time = 0.0;        // used to time binary resolution
double hr_start_time;         // used to time hyper-resolution
double hr_time = 0.0;         // used to time hyper-resolution
#endif // _STATS

#ifdef _EQVDEBUG
bool inEQVLIT = false;
#endif // _EQVDEBUG

#ifdef _CHECK
// True if the check functions of checkpreproc found any bugs.
extern bool foundBug;
#endif // _CHECK

// BUCKET STUFF
//bclsBucket * nextFreeBucket;
//bclsBucket ** freeBuckets;
//int numFreeBuckets;
//int freeBucketsArrayIndex;
//int freeBucketsChunkSize;
// END BUCKET STUFF

// These are variables for statistics gathering on the reduced theory.
// They are defined in the file cnfoutput.cpp
extern int NEW_NVARS;
extern int NEW_NLITS;
extern int NEW_NUMCLS;
extern int NEW_NUMBCLS;
extern int NEW_NUMNCLS;

// Horst: Added Prefix Information
// When reading a QBF Problem these arrays are filled with the correct data
extern int  *nArrPrefixLevel;
extern bool *bArrExistential;
// END


//#################### FUNCTION DECLARATIONS #######################//

// Declaration of functions defined in section one of this file.
void convertNaryToBcls(preNcls * Cls, int clsIndex);
void pushnewBclsStack(int L1, int L2);
void popnewBclsStack(int &L1, int &L2);

// From hypres.cpp
void runHypRes();

// TEMP SUBSUME
//void simpleSubsump();
//void kromSubsump();
// TEMP SUBSUME

// Declaration of functions defined in section three of this file.
void printReducedTheory();
void printImplications();
void fprintReducedTheory();
void fprintNaryIndices();
void fprintActiveLits();
void fprintLitValues();
void fprintEquivLits();
void fprintNaryOfLits();
void fprintImplications();
void fprintTimesVisited();

#ifdef _STATS
void printInitialStats();
void printReport();
#endif // _STATS


#ifdef _CHECK
// Declarations of the check functions from checkpreproc.cpp
void checkReducedTheory();
void checkDuplicate();
void checkImplications();
bool checkResolvable();
// BUCKET STUFF
//void checkBclsTables();
// BUCKET STUFF
#endif // _CHECK


//################### SECTION ONE: SUPPORT FUNCTIONS ##################//



inline int popnaryUPstack() {
  int L = naryUPstack->pop();
  int eqvL = getEquivLit(L);
  return(eqvL);
}

inline bool emptynaryUPstack() {
  return(naryUPstack->size == 0);
}

// The next five functions are used with the bclsUPstack which
// performs unit propagation on the binary clauses only.
void pushbclsUPstack(int L) {
  //Jessica: cannot set the value of a universal
  if(!isLitExistential(L)) {
    setContradiction();
    printf("c CONTRADICTION found! 9\n");
    return;
  }
  switch(getVal(L)) {
  case F:
    setContradiction();
    printf("c CONTRADICTION found! 4\n");
    break;
  case T:
    break;
  case U:
    if(bclsUPstack->member(negate(L))) {
      setContradiction();
      printf("c CONTRADICTION found! 5\n");
    }
    else if(!bclsUPstack->member(L)) {
      bclsUPstack->pushnew(L);
    }
    break;
  default:
    printf("ERROR: Illegal value for tval for literal %d.\n",
	   IntLitToExt(L));
    break;
  }
}

// The next five functions are used with the naryUPstack which
// performs unit propagation on the nary clauses only.
void pushnaryUPstack(int L) {
  //Jessica: cannot set the value of a universal
  if (!isLitExistential(L)) {
    setContradiction();
    printf("c CONTRADICTION found! 10\n");
    return;
  }
  switch(getVal(L)) {
  case F:
    setContradiction();
    printf("c CONTRADICTION found! 1\n");
    break;
  case T:
    break;
  case U:
    if(naryUPstack->member(negate(L))) {
      setContradiction();
      printf("c CONTRADICTION found! 2\n");
    }
    else if(!naryUPstack->member(L)) {
      naryUPstack->pushnew(L);
    }
    break;
  default:
    printf("ERROR: Illegal value for tval for literal %d.\n",
	   IntLitToExt(L));
    break;
  }
}


//Jessica: returns true if a contradiction was found
bool universalReduction(preNcls * Cls, int clsIndex) {
  inHypRes = false; 
  // If we are not in runHypRes(), then the clause must contain at least one
  // existential, but if we are in runHypRes() it might not.
  int nMaxExistentialPrefixLevel = inHypRes ? -1 : 0;
  int nMaxUniversalPrefixLevel = -1;
  int newlen = Cls->curLen;
  int curlen = inHypRes ? getHR_curLen(clsIndex) : Cls->curLen;
 
  for(int j = 0;j < curlen; j++) {
    //printf(" | %d %d %d", Cls->lits[j], getLitPrefixLevel(Cls->lits[j]), isLitExistential(Cls->lits[j]));
    // Existential
    if(isLitExistential(Cls->lits[j]))
    {
      if(getLitPrefixLevel(Cls->lits[j]) > nMaxExistentialPrefixLevel)
      {
        nMaxExistentialPrefixLevel = getLitPrefixLevel(Cls->lits[j]);
      }
    }
    else { // Universal
      if(getLitPrefixLevel(Cls->lits[j]) > nMaxUniversalPrefixLevel)
      {
        //printf(" HERE %d ", isLitExistential(Cls->lits[j]));
        nMaxUniversalPrefixLevel = getLitPrefixLevel(Cls->lits[j]);
      }
    }
  }
  //printf(" * %d %d", nMaxUniversalPrefixLevel, nMaxExistentialPrefixLevel);
  if(nMaxUniversalPrefixLevel > nMaxExistentialPrefixLevel)
  {
    // Find out the new length
    int unit = -1;
    for(int j = 0; j < curlen; j++) {
      if(inHypRes && getVal(Cls->lits[j])!=U) continue;
      if(getLitPrefixLevel(Cls->lits[j]) > nMaxExistentialPrefixLevel)
      {
        newlen--;
      }
      else {
        unit = Cls->lits[j];
      }
    }
    if(newlen==0) {
      printf("\n Empty Clause detected during universalReduction! \n");
      //assert(false);
      setContradiction();
      printf("c CONTRADICTION found! 8\n");
      return true;
    }
    else if(newlen == 1) {
      Cls->active = false;

      pushbclsUPstack(unit); // Should we use the equivLit?
      pushnaryUPstack(unit);
      if(CONTRADICTION.val)
        return true;  

      // Now, remove the clause's index from the list of nary clause indexes 
      // I think I need to use origLen here because it is the only time
      // it will be taken off the Ncls list??
      for(int i = 0; i < curlen; i++) {
        preLit * Lit = intToPreLit(Cls->lits[i]);
        for(int j = 0; j < Lit->numNcls; j++) {
          if(Lit->LitNcls[j] == clsIndex) {
            Lit->LitNcls[j] = Lit->LitNcls[--Lit->numNcls];
            break;
          }
        }
      }
    }

    for(int j = 0; j < curlen; j++) {
      if(inHypRes && getVal(Cls->lits[j]) != U) continue;
      // Remove the trailing universals
      if(getLitPrefixLevel(Cls->lits[j]) > nMaxExistentialPrefixLevel)
      {
        Cls->lits[j]=Cls->lits[curlen-1];
        curlen--;
        j--;
      }
    }
    Cls->curLen = curlen;
    assert(Cls->curLen == newlen);
  }
  
  return false;
}

// This function unit progates L through the nary theory.
void naryUnitProp(int L) {
  int i, j, k;
  preLit * Lit = intToPreLit(L);
  preLit * notLit = intToPreLit(negate(L));
  preLit * otherLit;
  preNcls * Cls;

  // Check if the literal has already been set.
  if(getNaryProp(Lit) && (getVal(Lit) == T))
    return;

  else if(getNaryProp(Lit) && (getVal(Lit) == F)) {
    setContradiction();
    printf("c CONTRADICTION found! 3\n");
    return;
  }

#ifdef _DEBUG
  printf("Unit propagating %d through nary clauses.\n", IntLitToExt(L));
#endif // _DEBUG

  // If the value of the literal was unknown, then set it true and set any
  // nary clauses that it appears in to inactive.  Also, remove the clause
  // from the LitNcls lists of all the other literals in the clause.
  setVal(Lit, T);
  setNaryProp(Lit, true);
  for(i = 0; i < getNumNcls(Lit); i++) {

#ifdef _STATS
    //temp-new
    //// Add one to the number of satisfied clauses if this clause was not
    //// already inactive.
    //if(getActive(getLitNclsElem(Lit, i)) == true) {
    numSatCls++;
    numSatNaryCls++;
    //}
#endif // _STATS

    setActive(getLitNclsElem(Lit, i), false);

    // Go through the other literals in this newly inactive clause and remove
    // the clause from their LitNcls list of nary clauses that they are in.
    for(j = 0; j < getCurLen(getLitNclsElem(Lit, i)); j++) {
      if(getLitsElem(getLitNclsElem(Lit, i), j) != L) {
	otherLit = intToPreLit(getLitsElem(getLitNclsElem(Lit, i), j));

	for(k = 0; k < otherLit->numNcls; k++) {
	  if(otherLit->LitNcls[k] == Lit->LitNcls[i]) {
	    otherLit->LitNcls[k] = otherLit->LitNcls[--otherLit->numNcls];
	    break;
	  }
	}
      }
    }
  }

  // Now set its negation to false and remove its negation from any clauses
  // it appears in.
  notLit->val = F;
  notLit->naryProp = true;
  for(i = 0; i < notLit->numNcls; i++) {
    int clsIndex = notLit->LitNcls[i];
    Cls = intToPreNcls(clsIndex);

    //temp-new
    //if(Cls->active) {

    // If the clauses is active, find the literal's negation and replace it with
    // the last literal in the clause and shorten the clause length.
    for(j = 0; j < Cls->curLen; j++) {
      if(Cls->lits[j] == negate(L)) {
	Cls->lits[j] = Cls->lits[--Cls->curLen];
	break;
      }
    }

    // Jessica: perform universal reduction
#ifndef _NO_UR
    if (universalReduction(Cls, clsIndex))
      return; 
#endif  
    
    // If the clauses has become binary, run the function convertNaryToBcls
    // to set it to inactive and add it to the binary clause stack.
    if(Cls->curLen == 2) {
      convertNaryToBcls(Cls, notLit->LitNcls[i]);
    }

    //temp-new
    //}
  }
}

inline void propagatenaryUPstack() {
  while(!CONTRADICTION.val && !emptynaryUPstack())
    naryUnitProp(popnaryUPstack());
  if(CONTRADICTION.val) {
    naryUPstack->empty();
    bclsUPstack->empty();
  }
}



inline int popbclsUPstack() {
  int L = bclsUPstack->pop();
  int eqvL = getEquivLit(L);
  return(eqvL);
}

inline bool emptybclsUPstack() {
  return(bclsUPstack->size == 0);
}

// This function unit progates L through the binary theory.
void bclsUnitProp(int L) {
  int i, j;
  preLit * Lit = intToPreLit(L);
  preLit * notLit = intToPreLit(negate(L));

  // Check if the literal has already been set.
  if((Lit->bclsProp) && (Lit->val == T))
    return;

  else if((Lit->bclsProp) && (Lit->val == F)) {
    setContradiction();
    printf("c CONTRADICTION found! 6\n");
    return;
  }

#ifdef _DEBUG
  printf("Unit propagating %d through binary clauses.\n", IntLitToExt(L));
#endif // _DEBUG

#ifdef _STATS
  numValuedLits += 2;
#endif // _STATS

  // If the value of the literal was unknown, then set it true and add any
  // literals implied by this literal to the UP stacks.
  Lit->val = T;
  Lit->bclsProp = true;

  for(i = 0; i < Lit->numBcls; i++) {
    // Jessica: if the resulting unit is universal, the contradiction will
    // be caught in pushbclsUPStack
    pushbclsUPstack(Lit->LitBcls[i]);
    pushnaryUPstack(Lit->LitBcls[i]);
    if(CONTRADICTION.val)
      return;

#ifdef _STATS
    if(!getBclsProp(Lit->LitBcls[i])) {
      numSatCls++;
      numSatBcls++;
    }
#endif // _STATS
  }

  // Now set its negation to false and remove implications involving
  // its negation from the other literals.
  // Jessica: this is just removing the satisfied clauses
  notLit->val = F;
  notLit->bclsProp = true;
  for(i = 0; i < notLit->numBcls; i++) {
    preLit * negOtherLit = intToPreLit(negate(notLit->LitBcls[i]));
    if(negOtherLit->val == U) {
      for(j = 0; j < negOtherLit->numBcls; j++) {
	if(negOtherLit->LitBcls[j] == L) {

	  // BUCKET STUFF
	  //deleteBclsFromTable(negOtherLit->num, L);
	  // END BUCKET STUFF

	  negOtherLit->LitBcls[j] = negOtherLit->LitBcls[--negOtherLit->numBcls];
	  break;
	}
      }
    }
#ifdef _STATS
    if(!negOtherLit->bclsProp) {
      numSatCls++;
      numSatBcls++;
    }
#endif // _STATS
  }
}

inline void propagatebclsUPstack() {
  while(!CONTRADICTION.val && !emptybclsUPstack()) {
//Jessica
    int lit = popbclsUPstack();
#ifdef _DEBUG_EQV_UR
    printf("calling bclsUnitProp(%d)\n", IntLitToExt(lit));
#endif
    bclsUnitProp(lit);
  }
  if(CONTRADICTION.val) {
    naryUPstack->empty();
    bclsUPstack->empty();
  }
}

// This function takes an nary clause that is left with two literals and
// converts it to a binary clause.  This requires the clause to set inactive
// and to be pushed onto the stack of new binary clauses to added to the
// binary theory at the appropriate time.  In addition, the clause must be
// removed from the remaining two literals' lists of nary clauses.
void convertNaryToBcls(preNcls * Cls, int clsIndex) {
  int i;

#ifdef _STATS
  // Increment the number of nary clauses converted to binary.
  numReducedNary++;
#endif // _STATS

  Cls->active = false;
  pushnewBclsStack(Cls->lits[0], Cls->lits[1]);

  // Now, remove the clause's index from the list of nary clause indexes
  // for the two remaining literals.
  preLit * Lit = intToPreLit(Cls->lits[0]);
  for(i = 0; i < Lit->numNcls; i++) {
    if(Lit->LitNcls[i] == clsIndex) {
      Lit->LitNcls[i] = Lit->LitNcls[--Lit->numNcls];
      break;
    }
  }
  Lit = intToPreLit(Cls->lits[1]);
  for(i = 0; i < Lit->numNcls; i++) {
    if(Lit->LitNcls[i] == clsIndex) {
      Lit->LitNcls[i] = Lit->LitNcls[--Lit->numNcls];
      break;
    }
  }
}

// Add the index of a clauses to a literal's list of indices
// of the clauses that it appears in.
void addNcls(int L, int c) {

  // Set up LitNcls for the first time if needed.
  if(getLitNcls(L) == NULL) {
    setMaxNcls(L, INITNclsPerLit);
    setLitNcls(L, new int[getMaxNcls(L)]);
    if(!getLitNcls(L))
      panic("Memory Allocation Failure #14P\n");
  }
  else if(getNumNcls(L) == getMaxNcls(L)) {

    // The four times multiplies was picked with some testing done
    // to reduce the number of delete calls.
    setMaxNcls(L, getMaxNcls(L) * 4);
    int * temp = new int[getMaxNcls(L)];
    if(!temp)
      panic("Memory Allocation Failure #15P\n");
    for(int i = 0; i < getNumNcls(L); i++)
      temp[i] = getLitNclsElem(L, i);
    // Must be removed in final version and replace by memory management.
    delete [] getLitNcls(L);

#ifdef _STATS
    deleteCount++;
#endif // _STATS

    setLitNcls(L, temp);
  }

  // First check if the implication already exists.
  int i;
  for(i = 0; i < getNumNcls(L); i++) {
    if(getLitNclsElem(L, i) == c) {
      printf("ERROR: Found clause index %d already in LitNcls of %d when adding it.\n",
	     c, IntLitToExt(L));
    }
  }

#ifdef _PARANOIA
  if(getNumNcls(L) >= getMaxNcls(L))
    printf("ERROR: The value of numNcls is larger than maxNcls for Lit %d", L);
#endif // _PARANOIA

  // Add the index and increment the number of indices.
  setLitNclsElem(L, getNumNcls(L), c);
  setNumNcls(L, getNumNcls(L) + 1);
}

// This functions adds an implication to the theory.  It must be called
// twice to add both halfs of a binary clause.  Perhaps there should
// be a function to add them at the same time.  Returns true if the
// clause was added and false if it was already there.
bool addBcls(int L, int impliedL) {

  // Set up LitBcls for the first time if needed.
  if(getLitBcls(L) == NULL) {
    setMaxBcls(L, INITBclsPerLit);
    setLitBcls(L, new int[getMaxBcls(L)]);
    if(!getLitBcls(L))
      panic("Memory Allocation Failure #2P\n");
  }

  else {
    // First check if the implication already exists.
    int i;


    for(i = 0; i < getNumBcls(L); i++) {
      if(getLitBclsElem(L, i) == impliedL) {

#ifdef _EQVDEBUG
	if(inEQVLIT) {
	  int l1 = 0, l2 = 0;
	  l1 = IntLitToExt(L);
	  l2 = IntLitToExt(impliedL);
	  printf("--Found %d->%d already there--", l1, l2);
	}
#endif // _EQVDEBUG

	return(false);
      }
    }


    // BUCKET STUFF
    //if(findBclsTable(L, impliedL))
    //return(false);
    // END BUCKET STUFF


    // If we reached here, the implication is not a duplicate.
    if(getNumBcls(L) == getMaxBcls(L)) {

      // The four times multiplies was picked with some testing done
      // to reduce the number of delete calls.
      setMaxBcls(L, getMaxBcls(L) * 4);
      int * temp = new int[getMaxBcls(L)];
      if(!temp)
	panic("Memory Allocation Failure #3P\n");
      for(i = 0; i < getNumBcls(L); i++)
	temp[i] = getLitBclsElem(L, i);
      // Must be removed in final version and replace by memory management.
      delete [] getLitBcls(L);

#ifdef _STATS
      deleteCount++;
#endif // _STATS

      setLitBcls(L, temp);
    }
  }

#ifdef _PARANOIA
  if(getNumBcls(L) >= getMaxBcls(L))
    printf("ERROR: The value of numNcls is larger than maxNcls for Lit %d", L);
#endif // _PARANOIA

  // Do the actual adding of the literal to LitBcls
  setLitBclsElem(L, getNumBcls(L), impliedL);
  setNumBcls(L, getNumBcls(L) + 1);

  // BUCKET STUFF
  //addBclsToTable(L, impliedL);
  // END BUCKET STUFF


  // Set the bclsAdded variable of the literal L to true to indicate that it
  // should be visited by the incrFindSCC() function.
  setBclsAdded(L, true);

#ifdef _EQVDEBUG
  int l1 = IntLitToExt(L);
  int l2 = IntLitToExt(impliedL);
  if(inEQVLIT) {
    printf(" + Added %d->%d to implications", l1, l2);
  }
  else
    printf("ADDBCLS: Added %d->%d to implications.\n", l1, l2);
#endif // _EQVDEBUG

  return(true);
}

// Puts new binary clauses on the newBclsStack.
void pushnewBclsStack(int L1, int L2) {
  if(newBclsStack->top == (newBclsStack->size - 1)) {
    newBclsStack->size *= 2;
    int * temp = new int[newBclsStack->size];
    if(!temp)
      panic("Memory Allocation Failure #4P\n");
    for(int i = 0; i <= newBclsStack->top; i++)
      temp[i] = newBclsStack->clauses[i];
    delete [] newBclsStack->clauses;

#ifdef _STATS
    deleteCount++;
#endif // _STATS

    newBclsStack->clauses = temp;
  }
  newBclsStack->clauses[++newBclsStack->top] = L1;
  newBclsStack->clauses[++newBclsStack->top] = L2;
}

// The function goes through the binary clauses generated from the nary
// clause theory.  It checks whether the clauses are already satisfied
// or have become unit clauses and deals with them accordingly.  For
// those clauses that are still binary, it runs the addBcls() function
// to add them to the binary theory.
// Jessica: we should also check for universal reduction
void addnewBclsStacktoLit() {
  int L1, L2;
  preLit * eqvL1;
  preLit * eqvL2;

#ifdef _STATS
#ifdef _DEBUG
  int test;
#endif // _DEBUG
#endif // _STATS

  while(!emptynewBclsStack()) {
    popnewBclsStack(L1, L2);

#ifdef _STATS
#ifdef _DEBUG
    test = 0;
#endif // _DEBUG
    numAddBcls++;
#endif // _STATS

    // First check if either literal in the binary clause has an equivalent
    // literal that has already been assigned a value.  If so, the clause
    // is no longer binary, and could be satisfied or cause a contradiction
    // or unit propagation.

    eqvL1 = intToPreLit(getEquivLit(L1));
    eqvL2 = intToPreLit(getEquivLit(L2));

#ifdef _STATS
    if(!((eqvL1->val == U) && (eqvL2->val == U))) {
      numSatCls++;
      numSatBcls++;
#ifdef _DEBUG
      test=1;
#endif // _DEBUG
    }
#endif // _STATS

    if((eqvL1->val == T) || (eqvL2->val == T))
      continue;
    else if(eqvL1->val == F) {
      if(eqvL2->val == F) {
	setContradiction();
	printf("c CONTRADICTION found! 7\n");
	return;
      }
      else {
	pushnaryUPstack(L2);
	pushbclsUPstack(L2);
	if(CONTRADICTION.val)
	  return;
      }
    }
    else if(eqvL2->val == F) {
      pushnaryUPstack(L1);
      pushbclsUPstack(L1);
      if(CONTRADICTION.val)
	return;
    }

    // Now check for the case where the two literals of the clause are
    // equivalent literals.  In this case, the equivalent literal is
    // forced.
    else if(eqvL1->num == eqvL2->num) {
      pushnaryUPstack(eqvL1->num);
      pushbclsUPstack(eqvL1->num);
      if(CONTRADICTION.val)
	return;

#ifdef _STATS
      numSatCls++;
      numSatBcls++;
#ifdef _DEBUG
      if(test == 1) {
	printf("TEST FAILED!! (A)\n");
	printf("Clause %d v %d failed test.\n", IntLitToExt(eqvL1->num),
	       IntLitToExt(eqvL2->num));
	printf("val(%d) = %d. val(%d) = %d.\n", IntLitToExt(eqvL1->num),
	       eqvL1->val, IntLitToExt(eqvL2->num), eqvL2->val);
      }
#endif // _DEBUG
#endif // _STATS

    }

    // Also check for the case where one literal's equivalent literal
    // is the negation of the other literal's equivalent literal.  In this
    // situation, the binary clause is already true.
    else if(eqvL1->num == negate(eqvL2->num)) {

#ifdef _STATS
      numSatCls++;
      numSatBcls++;
#ifdef _DEBUG
      if(test == 1) {
	printf("TEST FAILED!! (B)\n");
	printf("Clause %d v %d failed test.\n", IntLitToExt(eqvL1->num),
	       IntLitToExt(eqvL2->num));
	printf("val(%d) = %d. val(%d) = %d.\n", IntLitToExt(eqvL1->num),
	       eqvL1->val, IntLitToExt(eqvL2->num), eqvL2->val);
      }
#endif // _DEBUG
#endif // _STATS

      continue;
    }
    //Jessica
    else if(!isLitExistential(eqvL1->num) && getLitPrefixLevel(eqvL1) > getLitPrefixLevel(eqvL2)) {
      //If the clause is just two universals, the contradiction will be found
      pushnaryUPstack(eqvL2->num);
      pushbclsUPstack(eqvL2->num);
      if(CONTRADICTION.val)
        return;
    } 
    else if(!isLitExistential(eqvL2->num) && getLitPrefixLevel(eqvL2) > getLitPrefixLevel(eqvL1)) {
      pushnaryUPstack(eqvL1->num);
      pushbclsUPstack(eqvL1->num);
      if(CONTRADICTION.val)
        return;
    }
    // END Jessica

    // Otherwise, add the clause to the binary theory if it is not already
    // there.  Use equivalent literals in place of the actual literals in
    // the stack in case these literals are outdated.
    else if(addBcls(negate(eqvL1->num), eqvL2->num)) {
      addBcls(negate(eqvL2->num), eqvL1->num);

#ifdef _DEBUG
      printf("New BCLS is: %d v %d.\n", IntLitToExt(L1), IntLitToExt(L2));
#endif // _DEBUG

    }

#ifdef _STATS
    // If the binary clause could not be added, it is redundant and hence
    // satisfied.
    else {
      numSatCls++;
      numSatBcls++;
#ifdef _DEBUG
      if(test == 1) {
	printf("TEST FAILED!! (C)\n");
	printf("Clause %d v %d failed test.\n", IntLitToExt(eqvL1->num),
	       IntLitToExt(eqvL2->num));
	printf("val(%d) = %d. val(%d) = %d.\n", IntLitToExt(eqvL1->num),
	       eqvL1->val, IntLitToExt(eqvL2->num), eqvL2->val);
      }
#endif // _DEBUG
    }
#endif // _STATS

  }
}

//----------------------------------------------------------------------

// This function addes literals to the equivLitStack for removal from the theory.
void pushequivLitStack(int L) {

#ifdef _PARANOIA
  if(getVal(L) != U) {
    printf("ERROR: Equivalent literal being pushed has already been assigned a value.");
  }
#endif // _PARANOIA

  // Increase the stack size if necessary.
  if(equivLitStack->realTop == (equivLitStack->size - 1)) {
    equivLitStack->size *= 2;
    int * temp = new int[equivLitStack->size];
    if(!temp)
      panic("Memory Allocation Failure #5P\n");
    for(int i = 0; i <= equivLitStack->realTop; i++)
      temp[i] = equivLitStack->lits[i];
    delete [] equivLitStack->lits;

#ifdef _STATS
    deleteCount++;
#endif // _STATS

    equivLitStack->lits = temp;
  }
  equivLitStack->lits[++equivLitStack->realTop] = L;
}

// This function preforms reductions on the implication graph
// (binary clauses) based on the Strongly Connected Components
// (equivalent literals) found during the depth first seach.
void runbclsEquivLitReduction() {

  int L, i, j, index = 0;
  bool foundEqLit;

  // Process all the redundant literals on the equivalent literal stack.
  while(!emptyequivLitStack()) {
    L = popequivLitStack();
    // The literal popped from the stack.
    preLit * Lit = intToPreLit(L);
    // The other literal in Lit's binary clauses and its negation.
    preLit * otherLit;
    // The negation of the other literal.
    preLit * negOtherLit;

    // Go through all the implications of Lit and perform various reductions
    // depending on the relationship  between the two literals in the binary
    // clause.
    for(i = 0; i < Lit->numBcls; i++) {
      otherLit = intToPreLit(Lit->LitBcls[i]);
      negOtherLit = intToPreLit(negate(otherLit->num));

#ifdef _EQVDEBUG
      int l1 = IntLitToExt(L);
      int l2 = IntLitToExt(otherLit->num);
      int l3 = IntLitToExt(Lit->equivLit);
      int l4 = IntLitToExt(otherLit->equivLit);
      printf("Found implication %d->%d equivalent to %d->%d.\n", l1, l2, l3, l4);
#endif // _EQVDEBUG

      // If the two literals are equivalent (the literal implies its negation) OR
      // if one literal is equivalent to the other literal's negation (the literal
      // implies itself), the binary clause is now a unit or satisfied.  Either
      // way, the contrapostive implication is removed from the other literal's
      // negation's implication list.
      if((Lit->equivLit == negOtherLit->equivLit) ||
	 (Lit->equivLit == otherLit->equivLit)) {

	for(j = 0; j < negOtherLit->numBcls; j++) {
	  if(negOtherLit->LitBcls[j] == negate(L)) {

	    // BUCKET STUFF
	    //deleteBclsFromTable(negOtherLit->num, negate(L));
	    // END BUCKET STUFF

	    negOtherLit->LitBcls[j] = negOtherLit->LitBcls[--negOtherLit->numBcls];

#ifdef _EQVDEBUG
	printf(" - Removing %d->%d (1).\n", (-1 * l2), (-1 * l1));
	printf(" * This clauses is now satisfied.\n");
#endif // _EQVDEBUG

#ifdef _STATS
	numSatCls++;
	numSatBcls++;
#endif // _STATS

	    break;
	  }
	}

	// Also, if the two literals are equivalent (the literal implies its
	// negation), then their equivalent literal's negation is forced and
	// can be unit propagated.
	if(Lit->equivLit == negOtherLit->equivLit) {
	  pushnaryUPstack(negate(Lit->equivLit));
	  pushbclsUPstack(negate(Lit->equivLit));
	  if(CONTRADICTION.val)
	    return;

#ifdef _EQVDEBUG
	  printf(" - Unit propagating %d.\n", (-1 * l3));
#endif // _EQVDEBUG
	}
      }

#ifndef _NO_UR 
      // Jessica: consider universal reduction
      else if(!isLitExistential(Lit->equivLit) && getLitPrefixLevel(Lit->equivLit) > getLitPrefixLevel(otherLit->equivLit)) {
        //If the clause is just two universals, the contradiction will be found
        pushnaryUPstack(otherLit->equivLit);
        pushbclsUPstack(otherLit->equivLit);
        if(CONTRADICTION.val)
          return;
      }  
      else if(!isLitExistential(otherLit->equivLit) && getLitPrefixLevel(otherLit->equivLit) > getLitPrefixLevel(Lit->equivLit)) {
        pushnaryUPstack(negate(Lit->equivLit));
        pushbclsUPstack(negate(Lit->equivLit));
        if(CONTRADICTION.val)
          return;
      }
      // END Jessica
#endif // _NO_UR

      // Otherwise, the two literals are not equivalent and one is not equivalent
      // to the other's negation.  Add the other literal's equivalent literal
      // to equivLit's list of binary clauses.
      else {

#ifdef _STATS
	bool addSuccess =
#endif // _STATS

	addBcls(Lit->equivLit, otherLit->equivLit);

#ifdef _STATS
	if(addSuccess)
	  numAddBcls++;
#endif // _STATS

	// If the other literal has no equivalent literal, change the
	// contrapositive implication attached to the other literal's
	// negation to Lit's equivalent literal.
	if(otherLit->num == otherLit->equivLit) {

	  foundEqLit = false;
	  for(j = 0; j < negOtherLit->numBcls; j++) {
	    if(negOtherLit->LitBcls[j] == negate(L))
	      index = j;
	    else if(negOtherLit->LitBcls[j] == negate(Lit->equivLit))
	      foundEqLit = true;
	  }
	  if(foundEqLit) {

	    // BUCKET STUFF
	    //deleteBclsFromTable(negOtherLit->num, negate(L));
	    // END BUCKET STUFF

	    negOtherLit->LitBcls[index] = negOtherLit->LitBcls[--negOtherLit->numBcls];
	  }
	  else {
	    negOtherLit->LitBcls[index] = negate(Lit->equivLit);

	    // BUCKET STUFF
	    //deleteBclsFromTable(negOtherLit->num, negate(L));
	    //addBclsToTable(negOtherLit->num, negate(Lit->equivLit));
	    // END BUCKET STUFF


	    // Set the bclsAdded variable of negOtherLit to true to indicate that it
	    // should be visited by the incrFindSCC() function.
	    negOtherLit->bclsAdded = true;

#ifdef _EQVDEBUG
	    printf(" + Adding %d->%d.\n", (-1 * l2), (-1 * l3));
#endif // _EQVDEBUG

	  }

#ifdef _EQVDEBUG
	  printf(" - Removing %d->%d (2).\n", (-1 * l2), (-1 * l1));
#endif // _EQVDEBUG

#ifdef _STATS
	numSatCls++;
	numSatBcls++;
#endif // _STATS

	}

	// If the other literal has an equivalent literal, remove the contrapositive
	// implication from the other literal and add the implication to the
	// other literal's equivalent literal.
	else {

#ifdef _EQVDEBUG
	  printf("Found implication %d->%d equivalent to %d->%d.\n", l1, l2, l3, l4);
	  int found = false;
#endif // _EQVDEBUG

	  for(j = 0; j < negOtherLit->numBcls; j++) {
	    if(negOtherLit->LitBcls[j] == negate(L)) {

	      // BUCKET STUFF
	      //deleteBclsFromTable(negOtherLit->num, negate(L));
	      // END BUCKET STUFF

	      negOtherLit->LitBcls[j] = negOtherLit->LitBcls[--negOtherLit->numBcls];

#ifdef _EQVDEBUG
	      found = true;
#endif // _EQVDEBUG

#ifdef _STATS
	      numSatCls++;
	      numSatBcls++;
#endif // _STATS

	      break;
	    }
	  }

#ifdef _EQVDEBUG
	  if(!found)
	    printf("ERROR: Missing implication for binary clause!!\n");
#endif // _EQVDEBUG

	  addBcls(negOtherLit->equivLit, negate(Lit->equivLit));

#ifdef _EQVDEBUG
	  printf(" - Removing %d->%d (3).\n", (-1 * l2), (-1 * l1));
#endif // _EQVDEBUG

	}
      }
    }
  }
}


//----------------------------------------------------------------------

// This recursive function is the heart of the implication based binary
// resolution procedure.  The basic algorithm is as follows.  A depth-first
// search is run from all literals that have no antecedants in the
// implication graph.  A stack of the antecedants of the currently
// visited literal is maintained.  If the current literal's negation
// is on the stack, the current literal is marked for unit propagtion.
// Otherwise, the current literal is added to the antecedant list and
// the function recurses to the current literal's active children.  When
// the recursion returns, the current literal is removed from the
// antecedant stack.  This way, the antecedant stack always contains all
// the antecedant of the current literal.
void BinResDFS(preLit * L) {
  int i;
  preLit * child;

#ifdef _STATS
  DFScounter++;
  L->visitOrder++;
#endif // _STATS

  // Set visited to true so that runBinRes() doesn't start the recursion
  // at an already visited literal.
  L->visited = true;

  // Check if the L's negation is on the stack.  If so, the literal
  // is implied because the implication not(L)->L exists.  Thus add L
  // to the UPlist.
  if(getOnStack(negate(L->num))) {
    //printf("Setting %d to be UPed.\n", IntLitToExt(L->num));
    UPlist[L->num] = true;
  }

  // Otherwise, recursive run a depth first search on L's children.
  else {
    // Put this literal on the stack of antecedants of implications on this
    // branch of the DFS.
    L->onStack = true;

    // Run a DFS on all children (consequents) of this literal.
    for(i = 0; i < L->numBcls; i++) {
      child = intToPreLit(L->LitBcls[i]);

      // If the child or its negation has not been selected for unit
      // propagation, run the DFS on it.
      if(!UPlist[child->num] && !UPlist[negate(child->num)]) {
	//printf("Recursing to %d.\n", IntLitToExt(child->num));
	BinResDFS(child);
      }
    }
    //printf("Completed %d.\n", IntLitToExt(L->num));
    L->onStack = false;
  }
}

void runBinRes() {

  int i;

#ifdef _STATS
  int numUP = 0, visCount = 0;
  DFScounter = 0;
#endif // _STATS

  // Initialize the preLit fields required for the depth first search.
  for(i = 0; i < NLITS; i++) {
    if(active(i)) {
      UPlist[i] = false;
      setVisited(i, false);
      setOnStack(i, false);
    }

#ifdef _STATS
      setVisitOrder(i, 0);
#endif // _STATS

  }

  // Go through the active literals, and run a depth first search on all
  // the literals not yet encountered during a previous DFS.
  for(i = 0; i < NLITS; i++) {
    if(active(i) && !getVisited(i) && !UPlist[i]
       && (getNumBcls(negate(i)) == 0)) {
      //printf("->Running DFS on %d.\n", IntLitToExt(i));

#ifdef _STATS
      visCount++;
#endif // _STATS

      BinResDFS(intToPreLit(i));
    }
  }

#ifdef _STATS
  //printf("c The literals to be unit propagated are:\n");
  for(i = 0; i < NLITS; i++) {
    if(active(i) && (UPlist[i] == true)) {
      numUP++;
      //printf("%d ", IntLitToExt(i));
    }
  }
  //printf("\n");

  printf("c BinRes for loop visited %d literals.\n", visCount);
  printf("c DFScounter is %d.\n", DFScounter);
  printf("c BinRes found %d literals for UP.\n", numUP);
#endif // _STATS
}

//----------------------------------------------------------------------

//REMOVE
//----------------------------------------------------------------------
// Alternate version of Implication-based Binary Resolution which
// tries to save time by temporarily storing the transitive closure.

void BinResDFS2(preLit * L) {
  int i, impl;
  preLit * child;

#ifdef _STATS
  DFScounter++;
  L->visitOrder++;
#endif // _STATS

  L->visited = true;

  //printf("Visiting %d.\n", IntLitToExt(L->num));

  // Add L to its own implication list.
  L->implications->pushnew(L->num);

  // Iterate through L's children.  If a child or the child's negation has
  // already been put on the UPlist, skip this child.  Otherwise, check if
  // the child has been visited already.  If not, run BinResDFS2 on it now.
  // When BinResDFS2 returns, the child's implication list will have in it all
  // the literals implied by the child.  If the child was already visited,
  // the child's implication list already contains all literals implied by
  // this child.  Check if L's negation is implied by this child.
  // If so, add the negation of L to the UPlist.  Finally, if L's negation
  // was not found among the impications of the child, add all the
  // literals in the child's implication list to L's implication list.
  for(i = 0; i < L->numBcls; i++) {
    child = intToPreLit(L->LitBcls[i]);

    if(!UPlist[child->num] && !UPlist[negate(child->num)]) {

      if(!child->visited)
	BinResDFS2(child);

      if(child->implications->member(negate(L->num))) {
	UPlist[negate(L->num)] = true;
	//printf("Found %d->%d.\n", IntLitToExt(L->num), IntLitToExt(negate(L->num)));
      }

      // If the negation of L was not found on the child's implications list,
      // add all literals implied by the child to L's set of implications.
      // Because of the nature of UniqueSets, the elements of the child's
      // implication list must be removed and then copied back into the
      // child's implication list.
      else {
	while(!child->implications->isEmpty()) {
	  impl = child->implications->pop();
	  tempSet->pushnew(impl);
	  if(!L->implications->member(impl))
	    L->implications->pushnew(impl);
	}

	// Copy the child's implications back into the child's implication list.
	while(!tempSet->isEmpty())
	  child->implications->pushnew(tempSet->pop());
      }
    }
  }

  //printf("Completed visit to %d.\n", IntLitToExt(L->num));
}

void runBinRes2() {

  int i;

#ifdef _STATS
  int numUP = 0, visCount = 0;
  DFScounter = 0;
#endif // _STATS

  // Initialize the preLit fields required for the depth first search.
  for(i = 0; i < NLITS; i++) {
    if(active(i)) {
      UPlist[i] = false;
      setVisited(i, false);
      intToPreLit(i)->implications->empty();
    }

#ifdef _STATS
      setVisitOrder(i, 0);
#endif // _STATS

  }

  // Go through the active literals, and run a depth first search on all
  // the literals not yet encountered during a previous DFS.
  for(i = 0; i < NLITS; i++) {
    //if(active(i) && !getVisited(i) && !UPlist[i]) {
    if(active(i) && !getVisited(i) && !UPlist[i]
       && (getNumBcls(negate(i)) == 0)) {
      //printf("running DFS on %d.\n", IntLitToExt(i));

#ifdef _STATS
      visCount++;
#endif // _STATS

      BinResDFS2(intToPreLit(i));
    }
  }

#ifdef _STATS
  //printf("c The literals to be unit propagated are:\n");
  for(i = 0; i < NLITS; i++) {
    if(active(i) && (UPlist[i] == true)) {
      numUP++;
      //printf("%d ", IntLitToExt(i));
    }
  }
  //printf("\n");

  printf("c BinRes2 for loop visited %d literals.\n", visCount);
  printf("c DFScounter is %d.\n", DFScounter);
  printf("c BinRes2 found %d literals for UP.\n", numUP);
#endif // _STATS
}


//----------------------------------------------------------------------


//----------------------------------------------------------------------
// Third alternate version of Implication-based Binary Resolution which
// tries to save time by temporarily storing the transitive closure.

bool addImpl(preLit * L, int impliedL);

void BinResDFS3(preLit * L) {
  int i, j;
  bool foundNegation;
  preLit * child;

#ifdef _STATS
  DFScounter++;
  L->visitOrder++;
#endif // _STATS

  L->visited = true;

  //printf("Visiting %d.\n", IntLitToExt(L->num));

  // Add L to its own implication list.
  addImpl(L, L->num);

  // Iterate through L's children.  If a child or the child's negation has
  // already been put on the UPlist, skip this child.  Otherwise, check if
  // the child has been visited already.  If not, run BinResDFS3 on it now.
  // When BinResDFS3 returns, the child's implication list will have in it all
  // the literals implied by the child.  If the child was already visited,
  // the child's implication list already contains all literals implied by
  // this child.  Check if L's negation is implied by this child.
  // If so, add the negation of L to the UPlist.  Finally, if L's negation
  // was not found among the impications of the child, add all the
  // literals in the child's implication list to L's implication list.
  for(i = 0; i < L->numBcls; i++) {
    child = intToPreLit(L->LitBcls[i]);

    if(!UPlist[child->num] && !UPlist[negate(child->num)]) {

      if(!child->visited)
	BinResDFS3(child);

      foundNegation = false;
      for(j = 0; j < child->numImpl; j++) {
	if(child->implList[j] == negate(L->num)) {
	  UPlist[negate(L->num)] = true;
	  foundNegation = true;
	  //printf("Found %d->%d.\n", IntLitToExt(L->num), IntLitToExt(negate(L->num)));
	  break;
	}
      }

      // If the negation of L was not found on the child's implication list,
      // add all literals implied by the child to L's set of implications.
      if(!foundNegation) {
	for(j = 0; j < child->numImpl; j++)
	  addImpl(L, child->implList[j]);
      }
    }
  }
  //printf("Completed visit to %d.\n", IntLitToExt(L->num));

  //temp
#ifdef _CHECK
  //if((DFScounter % 500) == 499) {
  //checkDupl();
  //if(foundBug) {
  //fprintNaryOfLits();
  //exit(1);
  //}
  //}
#endif // _CHECK
}

void runBinRes3() {

  int i;

#ifdef _STATS
  int numUP = 0, visCount = 0;
  DFScounter = 0;
#endif // _STATS

  // Initialize the preLit fields required for the depth first search.
  for(i = 0; i < NLITS; i++) {
    if(active(i)) {
      UPlist[i] = false;
      setVisited(i, false);
      if(intToPreLit(i)->implList != NULL) {
	delete [] intToPreLit(i)->implList;
	intToPreLit(i)->implList = NULL;
      }
    }

#ifdef _STATS
      setVisitOrder(i, 0);
#endif // _STATS

  }

  // Go through the active literals, and run a depth first search on all
  // the literals not yet encountered during a previous DFS.
  for(i = 0; i < NLITS; i++) {
    //if(active(i) && !getVisited(i) && !UPlist[i]) {
    if(active(i) && !getVisited(i) && !UPlist[i]
       && (getNumBcls(negate(i)) == 0)) {
      //printf("running DFS on %d.\n", IntLitToExt(i));

#ifdef _STATS
      visCount++;
#endif // _STATS

      BinResDFS3(intToPreLit(i));
    }
  }

#ifdef _STATS
  //printf("c The literals to be unit propagated are:\n");
  for(i = 0; i < NLITS; i++) {
    if(active(i) && (UPlist[i] == true)) {
      numUP++;
      //printf("%d ", IntLitToExt(i));
    }
  }
  //printf("\n");
  printf("c BinRes3 for loop visited %d literals.\n", visCount);
  printf("c DFScounter is %d.\n", DFScounter);
  printf("c BinRes3 found %d literals for UP.\n", numUP);
#endif // _STATS
}

// Returns true if the implication was added and false if it was already
// there.
bool addImpl(preLit * L, int impliedL) {
  if(L->implList == NULL) {
    L->maxImpl = INITImplPerLit;
    L->implList = new int[L->maxImpl];
    L->numImpl = 0;
    if(!L->implList)
      panic("Memory Allocation Failure #18P\n");
  }

  else {
    // First check if the implication already exists.
    int i;
    for(i = 0; i < L->numImpl; i++) {
      if(L->implList[i] == impliedL) {
	return(false);
      }
    }
    // If we reached here the implication is not a duplicate.
    if(L->numImpl == L->maxImpl) {
      L->maxImpl *= 4;
      int * temp = new int[L->maxImpl];
      if(!temp)
	panic("Memory Allocation Failure #19P\n");
      for(i = 0; i < L->numImpl; i++)
	temp[i] = L->implList[i];
      // Must be removed in final version and replace by memory management.
      delete [] L->implList;

#ifdef _STATS
      deleteCount++;
#endif // _STATS

      L->implList = temp;
    }
  }

  L->implList[L->numImpl++] = impliedL;
  return(true);
}

//----------------------------------------------------------------------
//REMOVE


// This function simplifies the nary clauses based on the Strongly Connected
// Components (equivalent literals) found during the depth first seach.
void runnaryEquivLitReduction() {

  int L, i, j, index = 0;
  bool foundEqLit;
  preLit * Lit;
  preNcls * Cls;

  // Go through all redundant literals on the the equivalent literal stack
  // and simplify the nary clauses the literal appears in.
  while(!emptyequivLitStack()) {
    L = popequivLitStack();
    Lit = intToPreLit(L);


    // Check that the literal and its equivalent literal have not been
    // assigned a value.  If not, the clause should be simplified.
    if((Lit->val == U) && (getVal(Lit->equivLit) == U)) {

      // Iterate through the clauses of the literal.
      for(i = 0; i < Lit->numNcls; i++) {

	// Perhaps we should check if clauses are active?
	// For now we assume all inactive clauses of the literal
	// have been removed.
        int clsIndex = Lit->LitNcls[i];
	Cls = intToPreNcls(clsIndex);

	// There are three cases that must be dealt when trying to simplify
	// the clause:
	// CASE 1: The redundant literal is in the clause but its non-
	//   redundant equivalent literal is not.  Replace the redundant
	//   literal with its equivalent.
	// CASE 2: Both the redundant literal and its equivalent are in the
	//   clause.  Remove the redundant and leave the equivalent.
	// CASE 3: The redundant literal and the negation of the equivalent
	//   literal are in the clause.  The clause has become a tautology
	//   and can be remove from the theory.
	// Note: due to earlier simplifications, cases 2 and 3 should be
	//   mutualy exclusive.
	// In order to determine the case we are dealing with, we must search
	// through the clause and look for the redundant literal, its
	// equivalent literal, the negation of its equivalent literal.

	foundEqLit = false;
	for(j = 0; j < Cls->curLen; j++) {
	  if(Cls->lits[j] == L)
	    index = j;
	  else if(Cls->lits[j] == Lit->equivLit)
	    foundEqLit = true;
	  else if(Cls->lits[j] == negate(Lit->equivLit)) {
	    // This clause is case 3.  Remove the clause from the LitNcls list
	    // of all the literals in it, except for L since L will no longer be
	    // used anyway.  Then, set the clause to inactive.
	    int k;
	    preLit * Lit2;

	    for(j = 0; j < Cls->curLen; j++) {
	      if(Cls->lits[j] != L) {
		Lit2 = intToPreLit(Cls->lits[j]);

		for(k = 0; k < Lit2->numNcls; k++) {
		  if(Lit2->LitNcls[k] == Lit->LitNcls[i]) {
		    Lit2->LitNcls[k] = Lit2->LitNcls[--Lit2->numNcls];
		    break;
		  }
		}
	      }
	    }

	    Cls->active = false;

#ifdef _STATS
	    numSatCls++;
	    numSatNaryCls++;
#endif // _STATS

	    break;
	  }
	}
	// If the clause is still active, then it is case 1 or 2.
	if(Cls->active) {

	  // If the equivalent literal was found, it is case 2.  Thus,
	  // remove the redundant literal from the clause.
	  if(foundEqLit) {
	    Cls->lits[index] = Cls->lits[--Cls->curLen];

#ifndef _NO_UR
            // Jessica: also perform UR
            if(universalReduction(Cls, clsIndex))
              return;
#endif // _NO_UR

	    // If the clause has become binary, run convertNaryToBcls() to
	    // add the clause to the newBclsStack and remove it from the nary
	    // theory.
	    if(Cls->curLen == 2)
	      convertNaryToBcls(Cls, Lit->LitNcls[i]);
	  }

	  // Otherwise, it is case 1.  Change the redundant literal to
	  // its equivalent literal.  Add this clause to the equivalent
	  // literals list of clauses.
	  else {
	    Cls->lits[index] = Lit->equivLit;
// Jessica
#ifndef _NO_UR
            if(universalReduction(Cls, clsIndex))
              return;
#endif // _NO_UR
	    addNcls(Lit->equivLit, Lit->LitNcls[i]);
	  }
	}
      }
    }

    // If the literal or its equivalent already have a value, unit propagate L
    // or its negation (whichever is true) through the nary theory.
    else if(((Lit->val == T) || (getVal(Lit->equivLit) == T)) && !Lit->naryProp)
      naryUnitProp(L);
    else if(((Lit->val == F) || (getVal(Lit->equivLit) == F)) && !Lit->naryProp)
      naryUnitProp(negate(L));
  }
}
//----------------------------------------------------------------------

// BUCKET STUFF
/*
void initFreeBuckets() {
  freeBucketsChunkSize = initFreeBucketsChunkSize;
  freeBucketsArrayIndex = 0;
  freeBuckets = new (bclsBucket *) [freeBucketsArraySize];
  allocMoreFreeBuckets();
}
void allocMoreFreeBuckets() {
  if(freeBucketsArrayIndex > freeBucketsArraySize - 1) {
    printf("ERROR: freeBucket array size exceeded!\n");
    exit(1);
  }
  freeBuckets[freeBucketsArrayIndex] = new bclsBucket[freeBucketsChunkSize];
  numFreeBuckets = freeBucketsChunkSize;
  freeBucketsChunkSize *= ChunkSizeIncreaseFactor;
  nextFreeBucket = freeBuckets[freeBucketsArrayIndex++];
}

bclsBucket * getFreeBucket() {
  if(numFreeBuckets-- == 0)
    allocMoreFreeBuckets();
  return(nextFreeBucket++);
}

void initBclsTable(int L) {
  PRELITS[L].bclsTable = new bclsBucket[bclsTableSize];
  for(int i = 0; i < bclsTableSize; i++)
    PRELITS[L].bclsTable[i].lit = -1;
}

// Returns true if L2 was successfully added to L1's hashtable and
// false if L2 was already there.
bool addBclsToTable(int L1, int L2) {
  // A -1 indicates that this the first time anything was put into this
  // bucket.
  if(PRELITS[L1].bclsTable[L2 % bclsTableSize].lit == -1) {
    PRELITS[L1].bclsTable[L2 % bclsTableSize].lit = L2;
    PRELITS[L1].bclsTable[L2 % bclsTableSize].next = NULL;
    return(true);
  }
  // A -2 indicates that something was deleted from this spot in the
  // bucket.
  else if(PRELITS[L1].bclsTable[L2 % bclsTableSize].lit == -2) {
    PRELITS[L1].bclsTable[L2 % bclsTableSize].lit = L2;
    return(true);
  }
  else if(PRELITS[L1].bclsTable[L2 % bclsTableSize].lit != L2) {
    bclsBucket * temp = PRELITS[L1].bclsTable + (L2 % bclsTableSize);
    while(temp->next != NULL) {
      if(temp->next->lit == L2)
	return(false);
      else if(temp->next->lit == -2) {
	temp->next->lit = L2;
	return(true);
      }
      else
	temp = temp->next;
    }
    temp->next = getFreeBucket();
    temp->next->lit = L2;
    temp->next->next = NULL;
    return(true);
  }
  else
    return(false);
}

// Returns true if L2 is found in L1's hashtable of binary clauses.
bool findBclsTable(int L1, int L2) {
  if(PRELITS[L1].bclsTable[L2 % bclsTableSize].lit == -1)
    return(false);
  else if(PRELITS[L1].bclsTable[L2 % bclsTableSize].lit == L2)
    return(true);
  else {
    bclsBucket * temp = PRELITS[L1].bclsTable[L2 % bclsTableSize].next;
    while(temp != NULL) {
      if(temp->lit == L2)
	return(true);
      else
	temp = temp->next;
    }
    return(false);
  }
}

// Returns true if L2 is successfully deleted from L1's hashtable of binary
// clauses and false if not.
bool deleteBclsFromTable(int L1, int L2) {
  if(PRELITS[L1].bclsTable[L2 % bclsTableSize].lit == -1)
    return(false);
  else if(PRELITS[L1].bclsTable[L2 % bclsTableSize].lit == L2) {
    PRELITS[L1].bclsTable[L2 % bclsTableSize].lit = -2;
    return(true);
  }
  else {
    bclsBucket * temp = PRELITS[L1].bclsTable[L2 % bclsTableSize].next;
    while(temp != NULL) {
      if(temp->lit == L2) {
	temp->lit = -2;
	return(true);
      }
      else
	temp = temp->next;
    }
    return(false);
  }
}

void printBclsTables() {
  int i, j;
  printf("\nHere are the contents of the Bcls Tables\n");
  for(i = 0; i < NLITS; i++) {
    printf("#%d#\n", IntLitToExt(i));
    for(j = 0; j < bclsTableSize; j++) {
      if(PRELITS[i].bclsTable[j].lit != -1) {
	printf("    %d: %d (%d)", j, IntLitToExt(PRELITS[i].bclsTable[j].lit),
	       PRELITS[i].bclsTable[j].lit);
	bclsBucket * temp = PRELITS[i].bclsTable[j].next;
	while(temp != NULL) {
	  printf(" %d (%d)", IntLitToExt(temp->lit), temp->lit);
	  temp = temp->next;
	}
	printf("\n");
      }
    }
  }
}

void printBclsTable(int lit) {
  int j;
  printf("\nHere are the contents of the Bcls Table for %d\n",
	 IntLitToExt(lit)); {
    printf("#%d#\n", IntLitToExt(lit));
    for(j = 0; j < bclsTableSize; j++) {
      if(PRELITS[lit].bclsTable[j].lit != -1) {
	printf("    %d: %d (%d)", j, IntLitToExt(PRELITS[lit].bclsTable[j].lit),
	       PRELITS[lit].bclsTable[j].lit);
	bclsBucket * temp = PRELITS[lit].bclsTable[j].next;
	while(temp != NULL) {
	  printf(" %d (%d)", IntLitToExt(temp->lit), temp->lit);
	  temp = temp->next;
	}
	printf("\n");
      }
    }
    printf("%d:", IntLitToExt(lit));
    for(j = 0; j < getNumBcls(lit); j++)
      printf(" %d", IntLitToExt(getLitBclsElem(lit, j)));
    if(!active(lit)) {
      printf(" (INACTIVE)");
      printf(" Truth is %d\n", getVal(lit));
      printf("EquivLit = %d\n", IntLitToExt(getEquivLit(lit)));
    }
    printf("\n");
  }
}
*/
// END BUCKET STUFF


//################### SECTION TWO: PREPROCESSOR() ##################//

void preprocessor()
{
  int i, j, l1, l2;
  CLAUSELIST * clsptr;

  // BUCKET STUFF
  //initFreeBuckets();
  // END BUCKET STUFF


  // STEP 1: Initialize the PRELITS data structure and its member fields.
  PRELITS = new preLit[NLITS];
  if(!PRELITS)
    panic("Memory Allocation Failure #7P\n");
  for(i = 0; i < NLITS; i++) {
    setNum(i, i);
    setVal(i, U);
    setEquivLit(i, i);
    setLitBcls(i, NULL);
    setMaxBcls(i, 0);
    setNumBcls(i, 0);
    setLitNcls(i, NULL);
    setMaxNcls(i, 0);
    setNumNcls(i, 0);
    setNaryProp(i, false);
    setBclsProp(i, false);

    // Horst: Setting Prefix Type & Level using the data from readproblem.cpp
    setLitPrefixType(i, bArrExistential[i]);
    setLitPrefixLevel(i, nArrPrefixLevel[i]);
    // test
    //printf("\n Literal %d PL: %d Existential? %d", i, getLitPrefixLevel(i), isLitExistential(i));
    // END

    // BUCKET STUFF
    //initBclsTable(i);
    // END BUCKET STUFF


    //REMOVE
    //temp
    // Initialize the literal's set of implications to a size equal to
    // the number of literals.
    //intToPreLit(i)->implications = new UniqueSet(NLITS);
    //if(!intToPreLit(i)->implications)
    //panic("Memory Allocation Failure #16P\n");
    //REMOVE

  }

  // STEP 2: Initialize the PRENCLS data structure and add the active nary
  //         clauses to it.
  PRENCLS = new preNcls[NUMNCLS];
  if(!PRENCLS)
    panic("Memory Allocation Failure #8P\n");
  j = 0;
  for(i = 3; i < CLSGROUPS; i++) {
    for(clsptr = lenNclauses[i]; clsptr; clsptr = clsptr->pnext) {
      if(clsptr->active) {

#ifdef _PARANOIA
	if(j >= NUMNCLS)
	  printf("ERROR: More nclauses found then calculated in readProblem()");
#endif // _PARANOIA

#ifdef _DEBUG
        // Jessica: this was creating too much output
	//printf("Creating clause %d\n", j);
#endif // _DEBUG

	// Currently the actual cls from the lenNclauses is used at the lits in the
	// nary clause data structure.  Maybe a copy should be used instead?
	setLits(j, clsptr->cls);
	setCurLen(j, clsptr->clslen);
	setActive(j++, true);
      }
    }
  }

  // STEP 3: Add the indexes of nary clauses in the PRENCLS data structure
  //         to the LitNcls fields of PRELITS data structure.

  // Determine the number of nary clauses each literal is involved in by
  // looping through the PRENCLS data structure and counting all occurences
  // of each literal.
  for(i = 0; i < NUMNCLS; i++)
    for(j = 0; j < getCurLen(i); j++)
      setMaxNcls(getLitsElem(i, j), getMaxNcls(getLitsElem(i, j)) + 1);

  // Now store the indices of the nary clauses in the PRELITS data structure.
  for(i = 0; i < NUMNCLS; i++) {
    for(j = 0; j < getCurLen(i); j++) {
      preLit * L = intToPreLit(getLitsElem(i, j));
      if(L->numNcls == 0) {
	// Set the initial size of the LitNcls array to fourth times the number
	// of nary clauses currently involving this literal to allow room for
	// growth.
	L->maxNcls *= 4;
	L->LitNcls = new int[L->maxNcls];
	if(!L->LitNcls)
	  panic("Memory Allocation Failure #9P\n");
      }
#ifdef _PARANOIA
      if(L->numNcls >= L->maxNcls)
	printf("ERROR: More instances of literal %d in nary clauses then calculated", L->num);
#endif // _PARANOIA

      L->LitNcls[L->numNcls++] = i;
    }
  }

  // STEP 4: Put all unit clauses from the original theory onto the unit propagation
  //         stacks for the nary and binary clauses.

  // Initialize unit propagation stacks.
  naryUPstack = new UniqueSet(NLITS);
  bclsUPstack = new UniqueSet(NLITS);
  UPstack = new UniqueSet(NLITS);

  // Initialize newBclsStack with arbitrary initial size as the number of literals
  // in the theory.
  newBclsStack = new BCLSSTACKS();
  newBclsStack->size = (int)(NLITS * INITBclsStackSize);
  newBclsStack->clauses = new int[newBclsStack->size];
  if(!naryUPstack || !bclsUPstack || !UPstack || !newBclsStack
     || !newBclsStack->clauses)
    panic("Memory Allocation Failure #10P\n");
  newBclsStack->top = -1;

  // Put all the initial units into the appropriate UP stack.
  for(clsptr = lenNclauses[1]; clsptr; clsptr = clsptr->pnext) {

    //REMOVE
#ifdef _TWOUPSTACK
    pushnaryUPstack(clsptr->cls[0]);
    pushbclsUPstack(clsptr->cls[0]);
    if(CONTRADICTION.val)
      return;
#endif // _TWOUPSTACK
    //REMOVE

#ifdef _ONEUPSTACK
    pushUPstack(clsptr->cls[0]);
    if(CONTRADICTION.val)
      return;
#endif // _ONEUPSTACK

#ifdef _DEBUG
    printf("Adding %d to UP stack.\n", IntLitToExt(clsptr->cls[0]));
#endif // _DEBUG

#ifdef _STATS
    numSatCls++;
    numInitUnits++;
#endif // _STATS
  }

#ifdef _DEBUG
  // Jessica: this was creating too much output
  //printf("==>Here are the data structures before any UP.\n");
  //printReducedTheory();
#ifdef _STATS
  // Jessica: this was creating too much output, and is not even being
  // updated
  //printPreprocStats();
#endif // _STATS
#endif // _DEBUG


  // STEP 5: Unit propagate all unary clauses from the original theory through
  //         the nary clauses collecting and deactivating all clauses that become
  //         binary.

  //REMOVE
#ifdef _TWOUPSTACK
  propagatenaryUPstack();
  if(CONTRADICTION.val)
    return;
#endif // _TWOUPSTACK
  //REMOVE

#ifdef _DEBUG
  //printf("==>Here are the data structures after nary UP.\n");
  //printDS();
  //printnewBclsStack();
  //printReducedTheory();
#ifdef _STATS
   // Jessica: this was creating too much output and was not even being
   // updated
   //printPreprocStats();
#endif // _STATS
#endif // _DEBUG

  // STEP 6: Add all original and new generated binary clauses to the PRELITS
  //         data structure.

  // First, add the binary clauses from the original theory.
  // Determine the number of implications each literal is involved in by
  // counting the number of binary clauses containing the literal's negation
  // in the lenNclauses data structure.
  for(clsptr = lenNclauses[2]; clsptr; clsptr = clsptr->pnext) {
    setMaxBcls(negate(clsptr->cls[0]), getMaxBcls(negate(clsptr->cls[0])) + 1);
    setMaxBcls(negate(clsptr->cls[1]), getMaxBcls(negate(clsptr->cls[1])) + 1);

    NUMBCLS++;
  }

  // Store the original binary clauses into the PRELITS array using
  // the values calculated above for the size of the LitBlcs arrays.

  // This is used for checking for duplicate clauses.
  int found;

  for(clsptr = lenNclauses[2]; clsptr; clsptr = clsptr->pnext) {
    found = false;

    l1 = negate(clsptr->cls[0]);
    if(getNumBcls(l1) == 0) {
      // Set the initial size of the LitBcls array to four times the number of
      // implications that have l1 as the antecendant in order to leave room
      // for future growth.
      setMaxBcls(l1, getMaxBcls(l1) * 4);
      setLitBcls(l1, new int[getMaxBcls(l1)]);
      if(!getLitBcls(l1))
	panic("Memory Allocation Failure #11P\n");
    }
#ifdef _PARANOIA
    if(getNumBcls(l1) >= getMaxBcls(l1))
      printf("ERROR: More instances of literal %d in binary clauses then calculated\n", l1);
#endif // _PARANOIA

    // Make sure that this clause is not a duplicate.
    for(i = 0; i < getNumBcls(l1); i++) {
      if(getLitBclsElem(l1, i) == clsptr->cls[1]) {
	found = true;
	break;
      }
    }

    if(found) {

#ifdef _STATS
      numSatCls++;
      numSatBcls++;
#endif // _STATS

      // If it is a duplicate, don't add the other implication.
      continue;
    }

    // Do the actual adding of the implication to the data structure.
    setLitBclsElem(l1, getNumBcls(l1), clsptr->cls[1]);
    setNumBcls(l1, getNumBcls(l1) + 1);

    // BUCKET STUFF
    //addBclsToTable(l1, clsptr->cls[1]);
    // END BUCKET STUFF

    l2 = negate(clsptr->cls[1]);

    if(getNumBcls(l2) == 0) {
      // Set the initial size of the LitBcls array to four times the number of
      // implications that have l2 as the antecendant in order to leave room
      // for future growth.
      setMaxBcls(l2, getMaxBcls(l2)  * 4);
      setLitBcls(l2, new int[getMaxBcls(l2)]);
      if(!getLitBcls(l2))
	panic("Memory Allocation Failure #12P\n");
    }

#ifdef _PARANOIA
    if(getNumBcls(l2) >= getMaxBcls(l2))
      printf("ERROR: More instances of literal %d in binary clauses then calculated\n", l1);
#endif // _PARANOIA

    // Do the actual adding of the implication to the data structure.
    setLitBclsElem(l2, getNumBcls(l2), clsptr->cls[0]);
    setNumBcls(l2, getNumBcls(l2) + 1);

    // BUCKET STUFF
    //addBclsToTable(l2, clsptr->cls[0]);
    // END BUCKET STUFF
  }

  // BUCKET
  //printf("==>Here are the data structures after adding the bcls.\n");
  //printBclsTable(207);
  //printImplications();
  //checkBclsTables();
  // BUCKET

#ifdef _TWOUPSTACK
  // Now add the binary clauses derived during unit propagation of the nary
  // clauses to each literals list of implications.
  addnewBclsStacktoLit();
  if(CONTRADICTION.val)
    return;
#endif // _TWOUPSTACK

#ifdef _DEBUG
  // Jessica: this was creating too much output
  //printf("==>Here are the data structures after adding the bcls.\n");
  //printReducedTheory();
#ifdef _STATS
  //printPreprocStats();
#endif // _STATS
#endif // _DEBUG

  // STEP 7: Unit propagate all unary clauses from the original theory through
  //         the binary clauses for each literal.  Store newly valued literals
  //         on the unit progation stack for the nary clauses.
#ifdef _TWOUPSTACK
  propagatebclsUPstack();
  if(CONTRADICTION.val)
    return;
#endif // _TWOUPSTACK

#ifdef _DEBUG 
  // Jessica: This was creating too much output
  //printf("==>Here are the data structures after bcls UP.\n");
  //printReducedTheory();
#ifdef _STATS
  //printPreprocStats();
#endif // _STATS
#endif // _DEBUG

  //REMOVE
#ifdef _TWOUPSTACK
  // STEP 8: Repeat steps 5-7 with the newly valued literals from step 7
  //         instead of the original clauses.
  while(!emptynaryUPstack()) {
    propagatenaryUPstack();
    if(CONTRADICTION.val)
      return;

#ifdef _DEBUG
    // Jessica: this was creating too much output
    //printf("==>Here are the data structures after more nary UP.\n");
    //printnewBclsStack();
    //printReducedTheory();
#ifdef _STATS
    //printPreprocStats();
#endif // _STATS
#endif // _DEBUG

    // Add newly generated binary clauses.
    addnewBclsStacktoLit();
    propagatebclsUPstack();
    if(CONTRADICTION.val)
      return;

#ifdef _DEBUG
    // Jessica: this was creating too much output
    //printf("==>Here are the data structures after more bcls UP.\n");
    //printReducedTheory();
#ifdef _STATS
    //printPreprocStats();
#endif // _STATS
#endif // _DEBUG
  }
#endif //  _TWOUPSTACK
    //REMOVE

#ifdef _ONEUPSTACK
  // STEP 5, 7, and 8 NEW vestion.  Run propagateUPstack() to perform the same
  //         actions as the two old stacks did.
  propagateUPstack();
  if(CONTRADICTION.val)
    return;
#endif // _ONEUPSTACK

#ifdef _DEBUG
  // Jessica: this was creating too much output
  //printf("Before findSCC:\n");
  //printReducedTheory();
#endif // _DEBUG

  // BUCKET
  //printf("==>Here are the data structures after adding the bcls.\n");
  //printBclsTables();
  //printImplications();
  //printBclsTable(207);
  //printf("--2--\n");
  //checkBclsTables();
  // BUCKET



  // STEP 9: Use a Tarjan-like algorithm to find the Strongly Connected
  //         Components (SCCs) in the binary clause implication graph.
  //         The SCCs represent the sets of  equivalent literals in the
  //         theory.  Store the redundant literals on a stack for removal
  //         from the theory.

  // Initialize equivalent literal stack with arbitrary initial size for the
  // number of equivalent literals it can hold.
  equivLitStack = new EQVLITSTACKS;
  equivLitStack->lits = new int[NLITS / 10];
  if(!equivLitStack || !equivLitStack->lits)
    panic("Memory Allocation Failure #13P\n");
  equivLitStack->size = NLITS / 10;
  equivLitStack->realTop = -1;

#ifdef _STATS
  // Set the current number of literals to NLITS minus those already valued.
  CURNLITS = NLITS - numValuedLits;
#endif // _STATS

//Jessica
#ifdef _PERFORM_SCC 
  // Initialize and run SCC algorithm.
  initSCC();
  findSCC();

  if(CONTRADICTION.val)
    return;
  
  // Set each literals bclsAdded variable to false.  When ever a literal is the
  // antecendant of a newly added binary clause, bclsAdded is set to true.  When
  // the function incrFindSCC() is called, it only visits these literals and
  // any literals reached from them through implications.
  for(i = 0; i < NLITS; i++) {
    setBclsAdded(i, false);
  }

  // STEP 10: Use the equivalent literals to reduce the number of binary clauses
  //          in the theory and possibly produce some unit clauses.

  // Set temporary top to real top of stack.
  // The temporary top is used because later the stack will be iterated through
  // again for the nary clauses.
  equivLitStack->top = equivLitStack->realTop;
#ifdef _EQVDEBUG
  inEQVLIT = true;
#endif // _EQVDEBUG
  runbclsEquivLitReduction();
  //Jessica
  if(CONTRADICTION.val)
    return;

#ifdef _EQVDEBUG
  inEQVLIT = false;
#endif // _EQVDEBUG
#endif //PERFORM_SCC
  // BUCKET
  //printf("A\n");
  //printBclsTable(207);
  //if(active(207))
  //printf("%d IS ACTIVE!!\n", IntLitToExt(207));
  //else
  //printf("104 IS INACTIVE!!\n");
  // BUCKET

  // STEP 11: Do unit propagation on binary clauses using unit clauses generated
  //          by reducing the binary theory with equivalent literals.
  propagatebclsUPstack();
  if(CONTRADICTION.val)
    return;

#ifdef _DEBUG
  // Jessica: this was creating too much output
  //printf("After Bcls EquivLit Reduction:\n");
  //printReducedTheory();
#endif // _DEBUG

  //******* Binary Resolution is optional and set by the _BINRES flag *******//

#ifdef _BINRES
  // STEP 12: Perform binary resolution by running a depth first search on the
  //          binary implication graph on finding literals that imply their
  //          own negation.  The negation of these literals are force to be
  //          true and can be unit propagated.

  // Initialize a data structures to hold literals found to be forced.
  UPlist = new bool[NLITS];
  if(!UPlist)
    panic("Memory Allocation Failure #6P\n");

  //REMOVE
  //temp
  //tempSet = new UniqueSet(NLITS);
  //if(!tempSet)
  //panic("Memory Allocation Failure #17P\n");
  //REMOVE

#ifdef _STATS
  dfs_start_time = GetInternalRunTime();
#endif // _STATS
  runBinRes();
#ifdef _STATS
  dfs_time += (GetInternalRunTime() - dfs_start_time);
#endif // _STATS

  // Add literals on UPlists to unit propagation stacks for the nary and
  // binary clauses.
  for(i = 0; i < NLITS; i++) {
    if(active(i) && (UPlist[i] == true)) {
      pushnaryUPstack(i);
      pushbclsUPstack(i);
      if(CONTRADICTION.val)
	return;
    }
  }

  // BUCKET
  //printf("B\n");
  //printBclsTable(207);
  //if(active(207))
  //printf("%d IS ACTIVE!!\n", IntLitToExt(207));
  //else
  //printf("104 IS INACTIVE!!\n");
  // BUCKET

#ifdef _DEBUG
  // Jessica: this was creating too much output
  //printf("After BINRES:\n");
  //printReducedTheory();
#endif // _DEBUG

#endif // _BINRES

  // STEP 13: Unit propagate the literals from the UPlist through the
  //          binary theory and then unit propagate all the forced
  //          literals from step 9-12 through the nary clauses.
  propagatebclsUPstack();
  if(CONTRADICTION.val)
    return;

#ifdef _BINRES

  // This check can only be done here if binary resolution was performed.
#ifdef _CHECK
  if(!checkResolvable())
    fprintImplications();
#endif // _CHECK

#endif // _BINRES

#ifdef _DEBUG
  // Jessica: this was creating too much output
  //printf("After Bcls UP:\n");
  //printReducedTheory();
#endif // _DEBUG

  propagatenaryUPstack();
  if(CONTRADICTION.val)
    return;

#ifdef _DEBUG
  // Jessica: this was creating too much output
  //printf("After Nary UP:\n");
  //printReducedTheory();
#endif // _DEBUG

  //BUCKET
  //printf("C\n");
  //printBclsTable(207);
  //if(active(207))
  //printf("%d IS ACTIVE!!\n", IntLitToExt(207));
  //else
  //printf("104 IS INACTIVE!!\n");
  //BUCKET

//Jessica
#ifdef _PERFORM_SCC
  // STEP 14: Use the equivalent literals to simplify the nary clauses.  This
  //          may satisfy some nary clauses and convert others to binary clauses.

  // Reset temporary top to real top of stack.
  equivLitStack->top = equivLitStack->realTop;

#ifdef _EQVDEBUG
  inEQVLIT = true;
#endif // _EQVDEBUG

  runnaryEquivLitReduction();
  //Jessica
  if(CONTRADICTION.val)
    return;

#ifdef _EQVDEBUG
  inEQVLIT = false;
#endif // _EQVDEBUG

#ifdef _DEBUG
  // Jessica: this was creating too much output
  //printf("After Nary EquivLit Reduction:\n");
  //printReducedTheory();
#endif // _DEBUG
#endif //_PERFORM_SCC

  // STEP 15: Add the binary clauses generated during the simplification of the
  //          nary clauses to each literal's list of implications.  Also, unit
  //          propagate any unit clauses generated by removing equivalent literals
  //          from the nary theory.  Loop until there is nothing left to unit
  //          propagate.
  do {
    addnewBclsStacktoLit();
    propagatebclsUPstack();
    propagatenaryUPstack();
    if(CONTRADICTION.val)
      return;
  } while(!emptynewBclsStack());

#ifdef _DEBUG
  // Jessica: this was creating too much output
  //printf("After UP:\n");
  //printReducedTheory();
#endif // _DEBUG

  // BUCKET
  //printf("==>Here are the data structures after findscc.\n");
  //printBclsTables();
  //printImplications();
  //checkBclsTables();
  //printf("D\n");
  //printBclsTable(207);
  //if(active(207))
  //printf("%d IS ACTIVE-hypres!!\n", IntLitToExt(207));
  //else
  //printf("104 IS INACTIVE!!\n");
  // BUCKET

  // TEMP SUBSUME
  //simpleSubsump();
  //kromSubsump();
  //checkReducedTheory();
  // TEMP SUBSUME
#ifdef _DEBUG_EQV_UR
  //cnfoutput();
  //exit(0);
#endif
  // STEP 16: Perform Hyper-resolution to further reduce the theory.  In the
  //          process, new binary clauses may be generated, literals may be
  //          forced for unit propagation, and some binary clauses may be
  //          removed because they are redundant.

  // Set up an integer array needed by Hyper-resolution.
  // Initialize unit clause stack used to hold indices of clauses that
  // have become unit clauses.
  UnitPropStack = new int[NLITS];

  // Run Hyper-resolution.

#ifdef _STATS
  inHypRes = true;
  hr_start_time = GetInternalRunTime();
#endif // _STATS

  //Jessica
  inHypRes = true;

  //temp
  //do {
//Jessica
#ifdef _DEBUG_EQV_UR
  assert(bclsUPstack->size == 0);
#endif
  runHypRes();
  //if(CONTRADICTION.val)
  //return;
  //} while(foundNewForced);

#ifdef _STATS
  hr_time += (GetInternalRunTime() - hr_start_time);
  inHypRes = false;
#endif // _STATS

  //Jessica
  inHypRes = false;

  if(CONTRADICTION.val)
    return;

#ifdef _STATS
#ifdef _DETAILS
  iterNum = 1;
  printPreprocStats();
#endif // _DETAILS
#endif // _STATS

#ifdef _CHECK
  // Run functions to check the correctness of the theory.
  checkReducedTheory();
  checkDuplicate();
  checkImplications();
  // BUCKET
  //checkBclsTables();
  // BUCKET

#ifdef _DETAILS
  printf("\n");
#endif // _DETAILS
#endif // _CHECK

  // BUCKET
  //printf("E\n");
  //printBclsTable(207);
  //if(active(207))
  //printf("%d IS ACTIVE!!\n", IntLitToExt(207));
  //printf("--3--\n");
  //checkBclsTables();
  // BUCKET

  //%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//Jessica
#ifdef _DEBUG_EQV_UR
  int iternum = 0;
  
  //cnfoutput();
  //exit(0);
#endif

  // STEPS 9 through 16 are repeated as long as findSCC finds an equivalent
  // literals, BinRes finds a literals to set true, or HypRes finds a new
  // binary clause or forced literal.
  do {
//Jessica
#ifdef _DEBUG_EQV_UR
  iternum++;
  printf("iteration = %d\n", iternum);  
#endif
  
  foundNewUP = false;

//Jessica
#ifdef _PERFORM_SCC
  // STEP 9: Use a Tarjan-like algorithm to find the Strongly Connected
  //         Components (SCCs) in the binary clause implication graph.
  //         The SCCs represent the sets of  equivalent literals in the
  //         theory.  Store the redundant literals on a stack for removal
  //         from the theory.

  // Reset top of equivLitStack to -1;
  equivLitStack->realTop = -1;

#ifdef _STATS
  // Set the current number of literals to NLITS minus those already valued
  // and those that are redundant.
  CURNLITS = NLITS - numValuedLits - numEquivLits;
#endif // _STATS

  // Initialize and run the incremental SCC algorithm.
  initSCC();
  incrFindSCC();

  if(CONTRADICTION.val)
    return;

  // Set each literals bclsAdded variable to false.  When ever a literal is the
  // antecendant of a newly added binary clause, bclsAdded is set to true.  When
  // the function incrFindSCC() is called, it only visits these literals and
  // any literals reached from them through implications.
  for(i = 0; i < NLITS; i++) {
    setBclsAdded(i, false);
  }

  // STEP 10: Use the equivalent literals to reduce the number of binary clauses
  //          in the theory and possibly produce some unit clauses.

  // Set temporary top to real top of stack.
  // The temporary top is used because later the stack will be iterated through
  // again for the nary clauses.
  equivLitStack->top = equivLitStack->realTop;

#ifdef _EQVDEBUG
  inEQVLIT = true;
#endif // _EQVDEBUG

  runbclsEquivLitReduction();
  //Jessica
  if(CONTRADICTION.val)
    return;


#ifdef _EQVDEBUG
  inEQVLIT = false;
#endif // _EQVDEBUG
#endif //_PERFORM_SCC

  // STEP 11: Do unit propagation on binary clauses using unit clauses generated
  //          by reducing the binary theory with equivalent literals.
#ifdef _DEBUG_EQV_UR
  //cnfoutput();
  //exit(0);
#endif
  propagatebclsUPstack();
  if(CONTRADICTION.val)
    return;

  //******* Binary Resolution is optional and set by the _BINRES flag *******//

#ifdef _BINRES
  // STEP 12: Perform binary resolution by running a depth first search on the
  //          binary implication graph on finding literals that imply their
  //          own negation.  The negation of these literals are force to be
  //          true and can be unit propagated.

#ifdef _STATS
  dfs_start_time = GetInternalRunTime();
#endif // _STATS
  runBinRes();
#ifdef _STATS
  dfs_time += (GetInternalRunTime() - dfs_start_time);
#endif // _STATS

  // Add literals on UPlists to unit propagation stacks for the nary and
  // binary clauses.
  for(i = 0; i < NLITS; i++) {
    if(active(i) && (UPlist[i] == true)) {
      foundNewUP = true;
      pushnaryUPstack(i);
      pushbclsUPstack(i);
      if(CONTRADICTION.val)
	return;
    }
  }

#endif // _BINRES

  // STEP 13: Unit propagate the literals from the UPlist through the
  //          binary theory and then unit propagate all the forced
  //          literals from step 9-12 throught the nary clauses.
  propagatebclsUPstack();

#ifdef _BINRES

  // This check can only be done here if binary resolution was performed.
#ifdef _CHECK
  if(!checkResolvable())
    fprintImplications();
#endif // _CHECK

#endif // _BINRES

  propagatenaryUPstack();
  if(CONTRADICTION.val)
    return;
//Jessica
#ifdef _PERFORM_SCC
  // STEP 14: Use the equivalent literals to simplify the nary clauses.  This
  //          may satisfy some nary clauses and convert others to binary clauses.

  // Reset temporary top to real top of stack.
  equivLitStack->top = equivLitStack->realTop;

#ifdef _EQVDEBUG
  inEQVLIT = true;
#endif // _EQVDEBUG

  runnaryEquivLitReduction();
  //Jessica
  if(CONTRADICTION.val)
    return;

#ifdef _EQVDEBUG
  inEQVLIT = false;
#endif // _EQVDEBUG
#endif //_PERFORM_SCC

  // STEP 15: Add the binary clauses generated during the simplification of the
  //          nary clauses to each literal's list of implications.  Also, unit
  //          propagate any unit clauses generated by removing equivalent literals
  //          from the nary theory.  Loop until there is nothing left to unit
  //          propagate.
  do {
    addnewBclsStacktoLit();
    propagatebclsUPstack();
    propagatenaryUPstack();
    if(CONTRADICTION.val)
      return;
  } while(!emptynewBclsStack());

  // BUCKET
  //printf("F\n");
  //printBclsTable(207);
  //printf("--4--\n");
  //checkBclsTables();
  // BUCKET


  // TEMP SUBSUME
  //simpleSubsump();
  //kromSubsump();
  // TEMP SUBSUME

  // STEP 16: Perform Hyper-resolution to further reduce the theory.  In the
  //          process, new binary clauses may be generated, literals may be
  //          forced for unit propagation, and some binary clauses may be
  //          removed because they are redundant.

#ifdef _STATS
  inHypRes = true;
  hr_start_time = GetInternalRunTime();
#endif // _STATS

  //Jessica
  inHypRes = true;

  //temp
  //do {
//Jessica
#ifdef _DEBUG
  //cnfoutput();
  //exit(0);
#endif
  runHypRes();
  //if(CONTRADICTION.val)
  //return;
  //} while(foundNewForced);

#ifdef _STATS
  hr_time += (GetInternalRunTime() - hr_start_time);
  inHypRes = false;
#endif // _STATS

  //Jessica
  inHypRes = false;

  if(CONTRADICTION.val)
    return;


#ifdef _STATS
#ifdef _DETAILS
  iterNum++;
  //printPreprocStats();
#endif // _DETAILS
#endif // _STATS

#ifdef _CHECK
  // Run functions to check the correctness of the theory.
  checkReducedTheory();
  checkDuplicate();
  checkImplications();
  // BUCKET
  //checkBclsTables();
  // BUCKET
#ifdef _DETAILS
  printf("\n");
#endif // _DETAILS
#endif // _CHECK

  } while((equivLitStack->realTop > -1) || foundNewUP || foundNewBcls || foundNewForced);

  // TEMP SUBSUME
  //printReport();
  //temp
  //printf("Here is the initial reduced theory.\n");
  //printReducedTheory();
  //printf("Performing Simple Subsumption.\n");
  //simpleSubsump();
  //temp
  //printf("Here is the theory after simple subsumption.\n");
  //printReducedTheory();
  //printReport();
  //printf("Performing Krom Subsumption.\n");
  //kromSubsump();
  //temp
  //printf("Here is the theory after Krom subsumption.\n");
  //printReducedTheory();
  //fprintEquivLits();
  // END TEMP SUBSUME

#ifdef _CHECK
  // checkResolvable can only be run at the very end if BinRes is not used because
  // at no other point is there a guarantee that the resolutions checked have been
  // performed.
  if(!checkResolvable())
    fprintImplications();

  if(foundBug)
    printf("*** ERROR: Check functions found a bug! ***\n");

#ifdef _DETAILS
  else
    printf("c Check functions found no bugs.\n");
#endif // _DETAILS
#endif // _CHECK

#ifdef _STATS
#ifdef _DETAILS
  printf("c Total time spent running DFS was %.3f seconds.\n", dfs_time);
  printf("c Total time spent running HypRes was %.3f seconds.\n", hr_time);
#endif // _DETAILS
#endif // _STATS

  // Set the boolean flag SATISFIED to true if the preprocessor found a
  // variable assignment that satisfied all clauses in the theory.
  //if(numSatCls == (numInitUnits + NUMBCLS + NUMNCLS + numAddBcls - numReducedNary))
  //SATISFIED = true;

}

//################### SECTION THREE: DEBUGGING AND STATS ##################//

// This function prints the implications to standard out..
void printImplications() {
  int i, j;

  printf("\nHere is a list of the implications for each literal:\n");
  for(i = 0; i < NLITS; i++) {
    printf("%d:", IntLitToExt(i));
    for(j = 0; j < getNumBcls(i); j++)
      printf(" %d", IntLitToExt(getLitBclsElem(i, j)));
    if(!active(i))
      printf(" (INACTIVE)");
    printf("\n");
  }
  printf("--------------------------------------------------\n\n");
}


// This function basically dumps the contents of the data structures to
// the screen.
void printDS() {
  int i, j;

  printf("Here is a list of the nary clauses:\n");
  for(i = 0; i < NUMNCLS; i++) {
    printf("%d:", i);
    for(j = 0; j < getCurLen(i); j++)
      printf(" %d", IntLitToExt(getLitsElem(i, j)));
    if(!getActive(i))
      printf(" (INACTIVE)");
    printf("\n");
  }

  printf("\nHere are the indexes of the nary clauses each literal appears in:\n");
  for(i = 0; i < NLITS; i++) {
    printf("%d:", IntLitToExt(i));
    if(!getNaryProp(i)) {
      for(j = 0; j < getNumNcls(i); j++) {
	printf(" %d", getLitNclsElem(i, j));
      }
    }
    if(!active(i))
      printf(" (INACTIVE)");
    printf("\n");
  }

  printf("\nHere is a list of the implications for each literal:\n");
  for(i = 0; i < NLITS; i++) {
    printf("%d:", IntLitToExt(i));
    if(!getBclsProp(i)) {
      for(j = 0; j < getNumBcls(i); j++)
	printf(" %d", IntLitToExt(getLitBclsElem(i, j)));
    }
    if(!active(i)) {
      printf(" (INACTIVE)");
      if(getVal(i) == T)
	printf(" val = T ");
      else if(getVal(i) == F)
	printf(" val = F ");
      printf(" equivLit = %d", IntLitToExt(getEquivLit(i)));
    }
    printf("\n");
  }
}

// This function prints out the theory in cnf format, minus the trailing
// zeroes.
void printReducedTheory() {
  int i, j;

  printf("\nHere is the Reduced Theory:\n");

  for(i = 0; i < NLITS; i++) {
    if(!getBclsProp(i) && active(i)) {
      for(j = 0; j < getNumBcls(i); j++) {
	if(i < getLitBclsElem(i, j)) {
	  printf("%d %d\n", IntLitToExt(negate(i)), IntLitToExt(getLitBclsElem(i, j)));
	}
      }
    }
  }

  for(i = 0; i < NUMNCLS; i++) {
    if(getActive(i)) {
      for(j = 0; j < getCurLen(i); j++)
        printf("%d ", IntLitToExt(getLitsElem(i, j)));
      printf("\n");
    }
  }

  printf("---------------------------------------------\n");
}

// This function prints out the theory in cnf format, minus the trailing
// zeroes, to a file.  It also prints the literals and there equivalent
// literals and the indices of the nary clauses they belong to, to the
// file.
void fprintReducedTheory() {
  int i, j;
  //int convL, convL2;
  FILE * fp;
  char * filename = "theory.out";
  if((fp = fopen(filename, "w")) == NULL) {
    printf("ERROR: Failed to open %s.\n", filename);
  }


  fprintf(fp, "Here is the Reduced Theory:\n");

  for(i = 0; i < NLITS; i++) {
    if(active(i)) {
      for(j = 0; j < getNumBcls(i); j++) {
	if(i < getLitBclsElem(i, j)) {
	  fprintf(fp, "%d %d\n", IntLitToExt(negate(i)), IntLitToExt(getLitBclsElem(i, j)));
	}
      }
    }
  }

  for(i = 0; i < NUMNCLS; i++) {
    if(getActive(i)) {
      fprintf(fp, "%d: ", i);
      for(j = 0; j < getCurLen(i); j++)
        fprintf(fp, "%d ", IntLitToExt(getLitsElem(i, j)));
      fprintf(fp, "\n");
    }
  }
  /*
  fprintf(fp, "-------------------------------------------------------------------\n");
  fprintf(fp, "\nThe resulting equivalent literals are:\n");
  for(i = 0; i < NLITS; i++) {
    convL = IntLitToExt(i);
    convL2 = IntLitToExt(getEquivLit(i));
    fprintf(fp, "EquivLit for %d: %d; val = %d\n", convL, convL2, getVal(i));
  }

  fprintf(fp, "\nHere are the indices of the nary clauses each literal appears in:\n");
  for(i = 0; i < NLITS; i++) {
    fprintf(fp, "%d:", IntLitToExt(i));
    for(j = 0; j < getNumNcls(i); j++) {
      fprintf(fp, " %d", getLitNclsElem(i, j));
    }
    if(!active(i))
      fprintf(fp, " (INACTIVE)");
    fprintf(fp, "\n");
  }
  */
  fclose(fp);
}

// This function prints the index numbers of the active nary clauses to
// a file.
void fprintNaryIndices() {
  int i;
  FILE * fp;
  char * filename = "indices.out";
  if((fp = fopen(filename, "w")) == NULL) {
    printf("ERROR: Failed to open %s.\n", filename);
  }

  fprintf(fp, "Here are the Active Nary Indices:\n");
  for(i = 0; i < NUMNCLS; i++) {
    if(getActive(i)) {
      fprintf(fp, "%d\n", i);
    }
  }
  fclose(fp);
}

// This functions prints the number of the active literals to a file.
void fprintActiveLits() {
  int i;
  FILE * fp;
  char * filename = "activelits.out";
  if((fp = fopen(filename, "w")) == NULL) {
    printf("ERROR: Failed to open %s.\n", filename);
  }

  fprintf(fp, "Here are the Active Literals:\n");
  for(i = 0; i < NLITS; i++) {
    if(active(i)) {
      fprintf(fp, "%d\n", IntLitToExt(i));
    }
  }
  fclose(fp);
}

// This functions prints the literals and their value to a file.
void fprintLitValues() {
  int i, eqv;
  FILE * fp;
  char * filename = "litvalues.out";
  if((fp = fopen(filename, "w")) == NULL) {
    printf("ERROR: Failed to open %s.\n", filename);
  }

  fprintf(fp, "Here are the Literals and their Values:\n\n");
  for(i = 0; i < NLITS; i++) {
    fprintf(fp, "%d: ", IntLitToExt(i));
    eqv = i;
    while(eqv != getEquivLit(eqv))
      eqv = getEquivLit(eqv);
    if(getVal(eqv) == T)
      fprintf(fp, "T\n");
    else if(getVal(eqv) == F)
      fprintf(fp, "F\n");
    else
      fprintf(fp, "U\n");
  }
}

// This function prints the literals and their equivalents to a file.
void fprintEquivLits() {
  int i;
  FILE * fp;
  char * filename = "equiv.out";
  if((fp = fopen(filename, "w")) == NULL) {
    printf("ERROR: Failed to open %s.\n", filename);
  }

  fprintf(fp, "Here are the pairs of Equivalent Literals:\n\n");
  for(i = 0; i < NLITS; i++) {
    fprintf(fp, "%d: %d\n", IntLitToExt(i), IntLitToExt(getEquivLit(i)));
  }
  fclose(fp);
}

// This function prints the indices of the nary clauses that each literal
// appears in to a file.
void fprintNaryOfLits() {
  int i, j;
  FILE * fp;
  char * filename = "nary.out";
  if((fp = fopen(filename, "w")) == NULL) {
    printf("ERROR: Failed to open %s.\n", filename);
  }

  fprintf(fp, "Here are the indexes of the nary clauses each literal appears in:\n");
  for(i = 0; i < NLITS; i++) {
    fprintf(fp, "%d:", IntLitToExt(i));
    for(j = 0; j < getNumNcls(i); j++) {
      fprintf(fp, " %d", getLitNclsElem(i, j));
    }
    if(!active(i))
      fprintf(fp, " (INACTIVE)");
    fprintf(fp, "\n");
  }
  fclose(fp);
}

// This function prints the implications to a file.
void fprintImplications() {
  int i, j;
  FILE * fp;
  char * filename = "implicate.out";
  if((fp = fopen(filename, "w")) == NULL) {
    printf("ERROR: Failed to open %s.\n", filename);
  }

  fprintf(fp, "Here is a list of the implications for each literal:\n");
  for(i = 0; i < NLITS; i++) {
    fprintf(fp, "%d:", IntLitToExt(i));
    if(!getBclsProp(i)) {
      for(j = 0; j < getNumBcls(i); j++)
	fprintf(fp, " %d", IntLitToExt(getLitBclsElem(i, j)));
    }
    if(!active(i)) {
      fprintf(fp, " (INACTIVE)");
      if(getVal(i) == T)
	fprintf(fp, " T ");
      else if(getVal(i) == F)
	fprintf(fp, " F ");
      fprintf(fp, " equivLit = %d", IntLitToExt(getEquivLit(i)));
    }
    fprintf(fp, "\n");
  }
  fprintf(fp, "--------------------------------------------------\n\n");
  fclose(fp);
}

// This function prints the contents of the newBclsStack
void printnewBclsStack() {
  int i = 0, l1, l2;
  printf("\nThe contents of the newBclsStack are:\n");
  while(i <= newBclsStack->top) {
    l1 = IntLitToExt(newBclsStack->clauses[i++]);
    l2 = IntLitToExt(newBclsStack->clauses[i++]);
    printf("%d %d\n", l1, l2);
  }
}

#ifdef _STATS

// This function  prints the number of times each literal was visited by
// BinResDFS to a file.
void fprintTimesVisited() {
  int i, num = 0;
  FILE * fp;
  char * filename = "visited.out";
  if((fp = fopen(filename, "w")) == NULL)
    printf("ERROR: Failed to open %s.\n", filename);

  for(i = 0; i < NLITS; i++) {
    if((getNumBcls(negate(i)) == 0) && (getVisitOrder(i) > 1))
      num++;
  }
  fprintf(fp, "The number of top level lits visited multiple times is %d.\n", num);
  fprintf(fp, "Here is the number of times BinResDFS was called on each literal:\n");
  for(i = 0; i < NLITS; i++) {
    fprintf(fp, "%d: %d", IntLitToExt(i), getVisitOrder(i));
    if((getNumBcls(negate(i)) == 0) && (getVisitOrder(i) > 1))
      fprintf(fp, "**");
    fprintf(fp, "\n");
  }
  fclose(fp);
}

// This function prints information about the initial theory.
void printInitialStats() {
  printf("There were %d variables in the initial theory.\n", NLITS/2);
  printf("There were %d binary and %d nary clauses in the initial theory.\n",
	 NUMBCLS, NUMNCLS);
}

// This function prints a summary of the preprocessor's progress after every
// iteration of steps 9 to 16.
void printPreprocStats() {
  printf("c      -- ITERATION %d OF PREPROCESSOR --\n", iterNum);
  printf("c Preprocessor found %d unit clauses in original theory.\n", numInitUnits);
  printf("c Preprocessor found %d equivalent variables.\n", numEquivLits/2);
  printf("c Preprocessor generated %d new binary clauses.\n", numAddBcls - numReducedNary);
  printf("c Preprocessor reduced %d nary clauses to binary.\n", numReducedNary);
  printf("c Preprocessor satisfied %d clauses.\n", numSatCls);
  printf("c This includes %d nary and %d binary clauses.\n", numSatNaryCls, numSatBcls);
  printf("c Preprocessor assigned a value to %d variables.\n", numValuedLits/2);
  printf("c %d variables remain active and unassigned.\n",
	 (NLITS - numEquivLits - numValuedLits)/2);
  printf("c %d clauses remain active.\n\n", NUMNCLS + NUMBCLS + numInitUnits + numAddBcls
	 - numReducedNary - numSatCls);
  printf("c DeleteCount is %d.\n\n", deleteCount);
}

// This function prints a final summary report of the preprocessor's work.
void printReport() {
  printf("\nTEST FILE: %s\n", filename);
  printf("Initial number of variables: %d\n", NLITS/2);
  printf("Initial number of unit clauses: %d\n", numInitUnits);
  printf("Initial number of binary clauses: %d\n", NUMBCLS);
  printf("Initial number of nary clauses: %d\n", NUMNCLS);
  printf("Total number of equivalent variables found: %d\n", numEquivLits/2);
  printf("Number of variables assigned a value: %d\n", numValuedLits/2);
  printf("Number of new binary clauses generated: %d\n", numAddBcls - numReducedNary);
  printf("Number of nary clauses reduced to binary: %d\n", numReducedNary);
  printf("Number of satisfied binary clauses: %d\n", numSatBcls);
  printf("Number of satisfied nary clauses: %d\n", numSatNaryCls);
  printf("Total number of variables assigned or removed: %d\n", (numEquivLits + numValuedLits)/2);
  printf("Percentage of variables assigned or removed: %.1f",
	(((float) (numEquivLits + numValuedLits)) / NLITS) * 100.0);  putchar('%');
  printf("\nTotal number of nary clauses satisfied or reduced: %d\n",
	 numReducedNary + numSatNaryCls);
  printf("Percentage of nary clauses removed: %.1f",
	 (((float) (numReducedNary + numSatNaryCls)) / NUMNCLS) * 100.0); putchar('%');
  printf("\nNumber of binary clauses remaining in the theory: %d\n",
	 NUMBCLS + numAddBcls - numSatBcls);
  printf("Percentage of original binary clauses remaining: %.1f",
	 (((float) (NUMBCLS + numAddBcls - numSatBcls)) / NUMBCLS) * 100.0); putchar('%');
  printf("\nTotal number of satisfied clauses: %d\n", numSatCls);
  printf("Percentage of clauses satisfied: %.1f",
	 (((float) numSatCls) / (numInitUnits + NUMBCLS + NUMNCLS + numAddBcls - numReducedNary))
	 * 100.0); putchar('%');
  printf("\nRuntime: %.3f seconds\n\n", end_time - start_time);
}

#endif // _STATS


/*
void gatherStats() {
  int i, j, newNVARS = 0, newNUMBCLS = 0, newNUMNCLS = 0;
  bool * varInTheory = new bool[NLITS/2];

  for(i = 0; i < NLITS/2; i++)
    varInTheory[i] = false;

  // Go through the binary clauses, counting the number of remaining clauses and marking
  // all variables still in the theory.
  for(i = 0; i < NLITS; i++) {
    if(active(i)) {
      for(j = 0; j < getNumBcls(i); j++) {
	if(i < getLitBclsElem(i, j)) {
	  newNUMBCLS++;
	  varInTheory[variable_of(i)] = true;
	  varInTheory[variable_of(getLitBclsElem(i, j))] = true;
	}
      }
    }
  }

  // Go through the nary clauses, counting the number of active clauses and marking all
  // variables still in the theory.
  for(i = 0; i < NUMNCLS; i++) {
    if(getActive(i)) {
      newNUMNCLS++;
      for(j = 0; j < getCurLen(i); j++)
	varInTheory[variable_of(getLitsElem(i, j))] = true;
    }
  }

  // Go through the varInTheory mark array and count the number of variables still in
  // the theory.
  for(i = 0; i < NLITS; i++)
    if(varInTheory[i])
      newNVARS++;

  // If no binary or nary clauses remain in the theory set the SATISFIED flag to true.
  if((newNUMBCLS + newNUMNCLS) == 0)
    SATISFIED = true;

#ifdef _STATS
  // Run checks on the calculated values here.
  if(newNUMBCLS != (NUMBCLS + numAddBcls - numSatBcls))
    printf("ERROR: Problem in counting binary clauses.\n");
  if(newNUMNCLS != (NUMNCLS - (numReducedNary + numSatNaryCls)))
    printf("ERROR: Problem in counting nary clauses.\n");
#endif // _STATS

}
*/

void recordSummary() {
  FILE * fp;
#ifdef _UR_IN_HYPRES
  if((fp = fopen("preprocessor_b.dat", "a")) == NULL)
    printf("ERROR: Failed to open preprocessor_b.dat\n");
#else
  if((fp = fopen("preprocessor_a.dat", "a")) == NULL)
    printf("ERROR: Failed to open preprocessor_a.dat\n");
#endif

  fprintf(fp, "%s,", filename);
  if(CONTRADICTION.val)
    fprintf(fp, "0,");
  else if(SATISFIED)
    fprintf(fp, "1,");
  else
    fprintf(fp, "-1,");

  fprintf(fp, "%d,%d,%d,%d,%d,%d,%.3f\n", NLITS/2, NUMNCLS, NUMBCLS, NEW_NVARS,
	  NEW_NUMNCLS, NEW_NUMBCLS, end_time - start_time);
  fclose(fp);
}

void printSummary() {
  
  // Horst
  printf("%.3f", end_time - start_time);
  // Compact Output
  if(CONTRADICTION.val)
   printf(" UNSAT \n");
  else
    {
     if(SATISFIED)
      printf(" SAT \n");
     else
      printf(" UNKNOWN \n");
      }
  
  FILE * pFile;
  pFile = fopen ("result1","wt");
  if (pFile!=NULL)
  {
    if(CONTRADICTION.val)
     fputs("UNSAT", pFile); 
    else{ 
     if(SATISFIED)
      fputs("SAT",pFile);
     else
      fputs("UNKNOWN",pFile);
    }
    fclose (pFile);
  }

  /*
  int i;

  printf("c Preprocessor runtime was %.3f seconds.\n", end_time - start_time);
  printf("c \n");
  if(CONTRADICTION.val)
    printf("s UNSATISFIABLE\n");
  else if(SATISFIED) {
    printf("s SATISFIABLE\n");

    // Print out the assignment of variables that satisfies the cnf formula.
    printf("v ");
    for(i = 0; i < NLITS/2; i++) {
      if(var_assign[i])
	printf("%d ", orig_varnum[i] + 1);
      else
	printf("-%d ", orig_varnum[i] + 1);
    }
    printf("0\n");
    delete [] var_assign;
  }
  else {
    printf("c Satisfiability of theory still undetermined.\n");
    printf("c See file %s for simplified theory.\n", outfile);
    } */
}

