/*

PreQuel. Copyright Horst Samulowitz and Jessica Davies and Fahiem Bacchus 2006

 A QBF preprocessor based on performing extensive reasoning with the binary
 clause subtheory. 

Version 1.0

The software available at this site is copyright (c) 2006 by Horst Samulowitz 
Jessica Davies and Fahiem Bacchus. All right are reserved. 
Use of this software is permitted for non-commercial research purposes, and it may be copied only for that use. 
All copies must include this copyright message. This software and
any documentation and/or information supplied with it is distributed
on an as is basis. Horst Samulowitz and Jessica Davies and Fahiem Bacchus and the University of Toronto make no warranties, express or implied, including but not limited to implied warranties of merchantability and fitness for a particular purpose, regarding the documentation, functions or performance of such
software, documentation and/or information.

@misc{Samulowitz:Davies:Bacchus:2clsQ,
	author  = { Horst Samulowitz and Jessica Davies and Fahiem Bacchus },
	year    = { 2006 },
	title   = { A QBF preprocessor employing extensive binary clause reasoning },
	note    = { Available from
                    http://www.cs.toronto.edu/\~{}fbacchus/sat.html }
}

*/

//
// FILE: findscc.cpp
//

// OVERVIEW:
//
// This file contains the four function used to find strongly connected 
// components in the implication graph.  The first time the SCC algorithm
// is run, all active literals are visited.  However, in subsequent
// invocations, it is only run on literals for which a new implication
// was generated since the previous SCC function call.
// The algorithm used to computed the strongly-connected components is
// based on Tarjan's algorithm with the improvements developed by 
// Nuutila and Soisalon-Soininen, called "Algorithm 2" in the article,
// "On Finding the Strongly Connected Components in a Directed Graph".
//
// Jessica: Algorithm 2 of the paper was used for the SAT preprocessor,
// but Algorithm 1 must be used for the QBF preprocessor since all literals
// in a component must be known in order to check for: 
// 1. no component can contain more than one universal
// 2. if a component contains a universal, there cannot be an existential
//    at a lower prefix level in that component, and the universal becomes 
//    the root of the component

#include "preprocessor.h"


// Code in #indef _EQVDEBUG blocks is for debugging purposes. 
#ifdef _EQVDEBUG
FILE * progfp;
char * filename3 = "prog.out";
#endif // _EQVDEBUG

// The is the current number of literals that have been visited by visit()
// this time around.
int visitCount;

// A stack used by the SCC algorithm while finding all member of a
// component. 
int * LitStack;

// The top of LitStack.
int top;

#ifdef _STATS
// Both the following variable store the number of strongly connected 
// components found.  But the numbers are calculated differently for
// error checking.  This redundancy will be removed in the final
// version.
int EQVNLITS;
int numNewEquivLits;
#endif // _STATS

// The function initializes all the variables required for SCC and
// sets them to their default values.
void initSCC() {

#ifdef _EQVDEBUG
  if((progfp = fopen(filename3, "a")) == NULL) {
    printf("ERROR: Failed to open %s.\n", filename3);
  }
#endif // _EQVDEBUG

  int i;
  visitCount = 1;
  LitStack = new int[NLITS];
  if(!LitStack)
    panic("Memory Allocation Failure #1P\n"); 
  top = -1;

#ifdef _STATS
  EQVNLITS = 0;
  numNewEquivLits = 0;
#endif // _STATS

  for(i = 0; i < NLITS; i++) {

    // Set visited to false for all literals to allow the visited marker to 
    // represent which literals were looked at this time through findscc (they
    // will have visited = true) and which were not looked at because they
    // are inactive (they will have visited = false).
    setVisited(i, false);

    if(active(i)) {

      // Set visitOrder to be NLITS initially so that any literal actually 
      // assigned an order will be a lower value.
      setVisitOrder(i, NLITS);
      setInComponent(i, false);
      setOnStack(i, false);
      setLowestEquiv(i, -1);
    }
  }
}

// This recursive function performs the depth first traversal of the implication
// graph and using an efficient form of Tarjan's algorithm, it computes the
// strongly connected components which also represent the sets of equivalent
// literals.
void visit(preLit * L) {
  int i;
  preLit * child;

  L->visited = true;

  // Record order that literals were visited in.
  L->visitOrder = visitCount++;

  
#ifdef _EQVDEBUG
  fprintf(progfp, "Visiting %d as visit #%d (has %d edges)\n", IntLitToExt(L->num),
	  L->visitOrder, L->numBcls);
#endif // _EQVDEBUG
  
  // Visit all children of this node, depth first.
  for(i = 0; i < L->numBcls; i++) {
    child = intToPreLit(L->LitBcls[i]);
    
#ifdef _EQVDEBUG
    fprintf(progfp, "  Enter edge %d->%d\n", IntLitToExt(L->num),
	    IntLitToExt(child->num));
#endif // _EQVDEBUG
    
    // Visit any unvisited children.
    if(!child->visited) {

#ifdef _EQVDEBUG
      fprintf(progfp, " + Recurse to %d\n", IntLitToExt(child->num));
      if(!active(child))
	fprintf(progfp, "BUG!! eqv(%d) = %d\n", IntLitToExt(child->num),
		IntLitToExt(child->equivLit));
#endif // _EQVDEBUG

      visit(child);
    }

#ifdef _EQVDEBUG
    fprintf(progfp, " --  PRELITS[%d].equivLit=%d\n", IntLitToExt(child->num),
	    IntLitToExt(child->equivLit));
#endif // _EQVDEBUG

    // If these children are strongly connected to this node,
    // note that the corresponding literals are equivalent.
    // If the child is not already in a component, make L's
    // equivalent literal that of the childs if it was visited first.
    // Jessica: modified to follow Algorithm 1 
    if(!getInComponent(child->num)) {
      if(getVisitOrder(child->equivLit) < getVisitOrder(L->equivLit)) {

#ifdef _EQVDEBUG
        fprintf(progfp, " -- Changing equivLit of %d to %d\n", IntLitToExt(L->num),
		IntLitToExt(child->equivLit));
#endif
	// The equivLit of a literal is its "root" as described in the
	// algorithm.
        L->equivLit = child->equivLit;
      }
    }

#ifdef _EQVDEBUG
    fprintf(progfp, "  Exiting edge %d->%d\n", IntLitToExt(L->num),
	    IntLitToExt(child->num));
#endif // _EQVDEBUG

  }

  // If this node is a root of a connected component, process this component
  // by popping the final root candidates of the stack and setting their
  // inComponent variable true.  This prevents them from being looked at
  // again.  (A root is its own equivalent literal.)
  if(L->equivLit == L->num) {

#ifdef _STATS
    EQVNLITS++;
#endif // _STATS

// Jessica
    int universal = -1; 
    int lowestExistential = -1;
    int popLit;
    setInComponent(L->num, true); 
#ifdef _DEBUG_EQV_UR
    printf("Component with root = %d: {", IntLitToExt(L->num));
#endif
    if(!isLitExistential(L)) {
      universal = L->num;
    }
    else {
      lowestExistential = L->num;
    }
   
    while((top > -1) && (getVisitOrder(LitStack[top]) > L->visitOrder)) {
      popLit = LitStack[top--];
      setOnStack(popLit, false);
      if(!isLitExistential(popLit) && universal > -1) {
        setContradiction();
        printf("c CONTRADICTION found! two universals are equivalent\n");
        assert(false);
      } 
      else if (!isLitExistential(popLit)) {
        universal = popLit;
      }
      // Jessica: keep track of the lowest prefix level existential
      else if (lowestExistential < 0 || getLitPrefixLevel(popLit) < getLitPrefixLevel(lowestExistential)) {
        lowestExistential = popLit;
      } 
#ifdef _DEBUG_EQV_UR
      printf("%d, ", IntLitToExt(popLit));
#endif
      setInComponent(popLit, true);
    }
    // Jessica: Make sure that there is no existential with a lower prefix
    // level than the universal if there is one 
    if (universal > -1 && lowestExistential > -1 && getLitPrefixLevel(universal) > getLitPrefixLevel(lowestExistential)) {
      setContradiction();
      printf("c CONTRADICTION found! an existential at a lower prefix level is equivalent to a universal\n");
      assert(false);
    }
#ifdef _DEBUG_EQV_UR
    printf("}\n");
#endif
    if (universal > -1) {
      setLowestEquiv(L, universal);
    }
    else {
      setLowestEquiv(L, lowestExistential);
    }
  }
  else {
    LitStack[++top] = L->num;
  }

#ifdef _EQVDEBUG
  fprintf(progfp, "PRELITS[%d].equivLit=%d and inComp=%d\n",
	  IntLitToExt(L->num), IntLitToExt(L->equivLit), L->inComponent);
  fprintf(progfp, "End of visit to %d\n", IntLitToExt(L->num));
#endif // _EQVDEBUG

}

// This function is called by preprocessor() to run the SCC the first time.  It
// iterates through all the theory's literals making sure they all get visited.
void findSCC() {
  int i;

  // Go through the active literals visiting all of those not yet visited.
  for(i = 0; i < NLITS; i++) {
    if(active(i) && !getVisited(i))
      visit(intToPreLit(i));
  }

  // Make sure that each literal's equivalent literal is the root of its 
  // component which is not guaranteed by algorithm 2.  What is required
  // is to set a literals equivalent literal to the equivalent literal
  // of its current equivalent literal until these are the same literal.
  for(i = 0; i < NLITS; i++) {
    while(getEquivLit(i) != getEquivLit(getEquivLit(i)))
      setEquivLit(i, getEquivLit(getEquivLit(i)));
  }

  for(i = 0;i < NLITS; i++) {
    //Jessica: if the root has its lowestEquiv field filled in, it means
    // that it should be used as the root of the component instead
    if(getLowestEquiv(getEquivLit(i)) > -1) {
      setEquivLit(i, getLowestEquiv(getEquivLit(i)));
    }
  }

#ifdef _EQVDEBUG
  // Print out parts of the data structures relating to SCC.
  int convL, convL2;
  FILE *fp1, *fp2;
  char * filename1 = "bcls.out";
  char * filename2 = "equivlit.out";
  if((fp1 = fopen(filename1, "a")) == NULL) {
    printf("ERROR: Failed to open %s.\n", filename1);
  }
  if((fp2 = fopen(filename2, "a")) == NULL) {
    printf("ERROR: Failed to open %s.\n", filename2);
  }

  fprintf(fp1, "\nHere is the implication graph:\n");
  for(i = 0; i < NLITS; i++) {
    fprintf(fp1, "%d:", IntLitToExt(i));
    for(int j = 0; j < getNumBcls(i); j++)
      fprintf(fp1, " %d", IntLitToExt(getLitBclsElem(i, j)));
    fprintf(fp1, "\n");
  }

  fprintf(fp2, "\nThe resulting equivalent literals are:\n");
#endif // _EQVDEBUG

  // Do a few more things to each literals to finish up SCC.
  for(i = 0; i < NLITS; i++) {

    // Make the equivalent literals for the two literals of each variable,
    // the same by making both equal to the equivalent variable of lower
    // value.
    // Ex.  if equiv(1) = 2 and equiv(-1) = -3, make equiv(-1) = -2.
    if(getEquivLit(i) > negate(getEquivLit(negate(i))))
      setEquivLit(i, negate(getEquivLit(negate(i))));

#ifdef _EQVDEBUG
    convL = IntLitToExt(i);
    convL2 = IntLitToExt(getEquivLit(i));
    fprintf(fp2, "EquivLit for %d: %d; val = %d\n", convL, convL2, getVal(i));
#endif // _EQVDEBUG

    // Put all literals that are not the root of their component in the 
    // equivLitStack to be removed from the theory.
    if(getVisited(i) && (i != getEquivLit(i))) {
      pushequivLitStack(i);

#ifdef _EQVDEBUG
      fprintf(fp2, "@@\n");
#endif // _EQVDEBUG
#ifdef _STATS
      numNewEquivLits++;
#endif // _STATS

    }

#ifdef _EQVDEBUG
    // Check that a literal and its negation have the same equivalent variable.
    else if(getVisited(i) && (i == getEquivLit(i))) {
      if((i % 2) && !(getVisited(negate(i)) && (negate(i) == getEquivLit(negate(i))))) {
	fprintf(fp2, "ERROR\n");
	fprintf(fp2, "%d.visited = %d while %d.visited = %d.\n", IntLitToExt(i),
		getVisited(i), IntLitToExt(negate(i)), getVisited(negate(i)));
	if(getVisited(negate(i)))
	  fprintf(fp2, "TRUE\n");
	fprintf(fp2, "eqv(%d) = %d\n", IntLitToExt(negate(i)),
		IntLitToExt(getEquivLit(negate(i))));
	fprintf(fp2, "val(%d) = %d\n", IntLitToExt(i), getVal(i));
	fprintf(fp2, "val(%d) = %d\n", IntLitToExt(negate(i)), getVal(negate(i)));
      }
    }
#endif // _EQVDEBUG

    // Check each variables pair of literals to see if both literals
    // are in the same SCC (meaning that a literal is equivalent to
    // its negation).  This is a contradiction.
    // Only check each pair once: when i is even.
    // Jessica: added the getVisited check to fix a bug
    if(getVisited(i) && !(i%2) && (getEquivLit(i) == getEquivLit(negate(i)))) {
      setContradiction();
      printf("c CONTRADICTION found! A lit and its neg are in an scc\n");
      //printf("c %d is equivalent to %d.\n", IntLitToExt(i), (IntLitToExt(i) * -1));
      return;
    }
  }

#ifdef _EQVDEBUG
  fclose(fp1);
  fclose(fp2);
  fclose(progfp);
#endif // _EQVDEBUG

#ifdef _STATS
#ifdef _DETAILS
  printf("c FindSCC results are:\n");
  printf("c FindSCC visited %d variables.\n", (visitCount - 1)/2);
  printf("c FindSCC found %d equivalent variables\n", numNewEquivLits/2);
  printf("c FindSCC reduced theory from %d to %d variables.\n", CURNLITS/2, EQVNLITS/2);
#endif // _DETAILS
  numEquivLits += numNewEquivLits;
#endif // _STATS

  delete [] LitStack;
}

// The function is called by the preprocessor() to run SCC after the first time.
// It only visits literals, which have had a new binary clause added to their list,
// and their children
void incrFindSCC() {
  int i;

  // Go through the active literals visiting all of those not yet visited.
  for(i = 0; i < NLITS; i++) {
    if(active(i) && getBclsAdded(i) && !getVisited(i))
      visit(intToPreLit(i));
  }

  // Make sure that each literal's equivalent literal is the root of its 
  // component which is not guaranteed by algorithm 2.  What is required
  // is to set a literals equivalent literal to the equivalent literal
  // of its current equivalent literal until these are the same literal.
  for(i = 0; i < NLITS; i++) {
    while(getEquivLit(i) != getEquivLit(getEquivLit(i)))
      setEquivLit(i, getEquivLit(getEquivLit(i)));
  }

  for(i = 0; i < NLITS; i++) {
    //Jessica: if the root has its lowestEquiv field filled in, it means
    // that it should be used as the root of the component instead
    if(getLowestEquiv(getEquivLit(i)) > -1) {
      setEquivLit(i, getLowestEquiv(getEquivLit(i)));
    }
  }

#ifdef _EQVDEBUG
  // Print out parts of the data structures relating to SCC.
  int convL, convL2;
  FILE *fp1, *fp2;
  char * filename1 = "incr_bcls.out";
  char * filename2 = "incr_equivlit.out";
  if((fp1 = fopen(filename1, "a")) == NULL) {
    printf("ERROR: Failed to open %s.\n", filename1);
  }
  if((fp2 = fopen(filename2, "a")) == NULL) {
    printf("ERROR: Failed to open %s.\n", filename2);
  }

  fprintf(fp1, "\nHere is the implication graph:\n");
  for(i = 0; i < NLITS; i++) {
    fprintf(fp1, "%d:", IntLitToExt(i));
    for(int j = 0; j < getNumBcls(i); j++)
      fprintf(fp1, " %d", IntLitToExt(getLitBclsElem(i, j)));
    fprintf(fp1, "\n");
  }

  fprintf(fp2, "\nThe resulting equivalent literals are:\n");
#endif // _EQVDEBUG

  for(i = 0; i < NLITS; i++) {

    // Make the equivalent literals for the two literals of each variable,
    // the same by making both equal to the equivalent variable of lower
    // value.
    // Ex.  if equiv(1) = 2 and equiv(-1) = -3, make equiv(-1) = -2.
    if(getEquivLit(i) > negate(getEquivLit(negate(i))))
      setEquivLit(i, negate(getEquivLit(negate(i))));

#ifdef _EQVDEBUG
    convL = IntLitToExt(i);
    convL2 = IntLitToExt(getEquivLit(i));
    fprintf(fp2, "EquivLit for %d: %d; val = %d\n", convL, convL2, getVal(i));
#endif // _EQVDEBUG

    // Put all literals that are not the root of their component in the 
    // equivLitStack to be removed from the theory.
    if((getVisited(i) || getVisited(negate(i)))
       && (i != getEquivLit(i))) {
      pushequivLitStack(i);

#ifdef _EQVDEBUG
      fprintf(fp2, "@@\n");
#endif // _EQVDEBUG
#ifdef _STATS
      numNewEquivLits++;
#endif // _STATS

    }

#ifdef _EQVDEBUG
    else if(getVisited(i) && (i == getEquivLit(i))) {
     // Check that a literal and its negation have the same equivalent variable.
      if((i % 2) && !(getVisited(negate(i)) && (negate(i) == getEquivLit(negate(i))))) {
	fprintf(fp2, "ERROR\n");
	fprintf(fp2, "%d.visited = %d while %d.visited = %d.\n", IntLitToExt(i),
		getVisited(i), IntLitToExt(negate(i)), getVisited(negate(i)));
	if(getVisited(negate(i)))
	  fprintf(fp2, "TRUE\n");
	fprintf(fp2, "eqv(%d) = %d\n", IntLitToExt(negate(i)),
		IntLitToExt(getEquivLit(negate(i))));
	fprintf(fp2, "val(%d) = %d\n", IntLitToExt(i), getVal(i));
	fprintf(fp2, "val(%d) = %d\n", IntLitToExt(negate(i)), getVal(negate(i)));
      }
    }
#endif // _EQVDEBUG


    // Check each variables pair of literals to see if both literals
    // are in the same SCC (meaning that a literal is equivalent to
    // its negation).  This is a contradiction.
    // Only check each pair once: when i is even.
    //Jessica: added the getVisited
    if(getVisited(i) && !(i % 2) && (getEquivLit(i) == getEquivLit(negate(i)))) {
      setContradiction();
      printf("c CONTRADICTION found! a lit and its neg are in an scc\n");
      //printf("c %d is equivalent to %d.\n", IntLitToExt(i), (IntLitToExt(i) * -1));
      return;
    }
  }

#ifdef _EQVDEBUG
  fclose(fp1);
  fclose(fp2);
  fclose(progfp);
#endif // _EQVDEBUG

#ifdef _STATS
#ifdef _DETAILS
  printf("c IncrFindSCC results are:\n");
  printf("c IncrFindSCC visited %d variables.\n", (visitCount - 1)/2);
  printf("c IncrFindSCC found %d equivalent variables\n", numNewEquivLits/2);
  printf("c IncrFindSCC reduced theory from %d to %d variables.\n", CURNLITS/2,
	 (CURNLITS - numNewEquivLits)/2);
#endif // _DETAILS
  numEquivLits += numNewEquivLits;
#endif // _STATS

  delete [] LitStack;
}
