//
// FILE:  checkpreproc.cpp
//

// OVERVIEW:
//
// This file contains 4 functions that check the contents of the
// preprocessor data structure to make sure they are consistent and
// do not contain any literals or clauses that should not be there.

#include "preprocessor.h"

// Flag indicating if any bugs were found during all check functions.
bool foundBug = false;

// This function counts the number of active literals in the PRELITS
// data structure.  It also counts the total number of binary clauses
// in the data structure.  In addition, it counts the number of
// active nary clauses in the PRENCLS data structure.  Finally, it
// checks all implications and nary clauses for literals that are
// inactive.  There should be NO inactive literals in any of the
// remaining implications and nary clauses.
void checkReducedTheory() {
  int i, j, numActiveImpl = 0, numActiveBcls = 0, numActiveNary = 0;
  int numActiveLit = 0, numInActiveLit = 0, numInActiveNary = 0;

  for(i = 0; i < NLITS; i++) {
    if(active(i)) {
      numActiveLit++;
      numActiveImpl = numActiveImpl + getNumBcls(i);
      for(j = 0; j < getNumBcls(i); j++) {
	if(!active(getLitBclsElem(i, j)))
	  numInActiveLit++;
      }
      for(j = 0; j < getNumNcls(i); j++) {
        //printf("\n i: %d j: %d", i, j);
        //assert(false);
	if(!getActive(getLitNclsElem(i, j))) {
	  numInActiveNary++;

	  //temp
	  printf("Inactive clause index %d in literal %d:\n", getLitNclsElem(i, j),
		 IntLitToExt(i));
	  for(int k = 0; k < getCurLen(getLitNclsElem(i, j)); k++)
	    printf("%d ", IntLitToExt(getLitsElem(getLitNclsElem(i, j), k)));
	  printf("\n");
	}
      }
    }
  }
  if((numActiveImpl % 2) == 1)
    printf("ERROR: There are an odd number of implications.\n");
  numActiveBcls = numActiveImpl / 2;

  for(i = 0; i < NUMNCLS; i++) {
    if(getActive(i)) {
      numActiveNary++;
      for(j = 0; j < getCurLen(i); j++) {
        if(!active(getLitsElem(i, j)))
	  numInActiveLit++;
      }
    }
  }

#ifdef _DETAILS
  printf("Reduced Theory has %d active variables.\n", numActiveLit/2);
  printf("Reduced Theory contains %d inactive variables.\n", numInActiveLit/2);
  printf("Reduced Theory has %d active binary clauses.\n", numActiveBcls);
  printf("Reduced Theory has %d active nary clauses.\n", numActiveNary);
  printf("Reduced Theory contains %d inactive nary clauses.\n", numInActiveNary);
  printf("Reduced Theory has %d active clauses in total.\n", numActiveBcls
	 + numActiveNary);
#endif // _DETAILS

  if((numInActiveLit > 0) || (numInActiveNary > 0)) {
    printf("ERROR: Found inactive literals or nary clauses in the theory!\n");
    foundBug = true;
  }
}

// This function checks for various duplications in the data structures.
// It checks the PRELITS data structure to make sure no literal has
// two or more of the same implication in the LitBcls array.  It checks
// that all the indices of clauses in the LitNcls array are distinct.
// Finally it checks all nary clauses to make sure they do not have
// duplicates of a literal.
void checkDuplicate() {
  bool * LitFound = new bool[NLITS];
  bool * IndexFound = new bool[NUMNCLS];
  bool foundDuplicate = false;
  int i, j;
  for(i = 0; i < NLITS; i++)
    LitFound[i] = false;
  for(i = 0; i < NUMNCLS; i++)
    IndexFound[i] = false;

  // Check for duplicate implications.
  for(i = 0; i < NLITS; i++) {
    for(j = 0; j < getNumBcls(i); j++) {
      if(LitFound[getLitBclsElem(i, j)]) {
	printf("ERROR: Found duplicate literal %d in implications of %d!\n",
	       IntLitToExt(getLitBclsElem(i, j)), IntLitToExt(i));
	foundDuplicate = true;
      }
      LitFound[getLitBclsElem(i, j)] = true;
    }
    for(j = 0; j < getNumBcls(i); j++)
      LitFound[getLitBclsElem(i, j)] = false;

    // Check for duplicate clause indices.
    for(j = 0; j < getNumNcls(i); j++) {
      if(IndexFound[getLitNclsElem(i, j)]) {
	printf("ERROR: Found duplicate index %d in nary clauses of %d!\n",
	       getLitNclsElem(i, j), IntLitToExt(i));
	foundDuplicate = true;
      }
      IndexFound[getLitNclsElem(i, j)] = true;
    }
    for(j = 0; j < getNumNcls(i); j++)
      IndexFound[getLitNclsElem(i, j)] = false;
  }

  // Check for duplicate literals in nary clauses.
  for(i = 0; i < NUMNCLS; i++) {
    for(j = 0; j < getCurLen(i); j++) {
      if(LitFound[getLitsElem(i, j)]) {
	printf("ERROR: Found duplicate literal %d in nary clause %d!\n",
		 IntLitToExt(getLitsElem(i, j)), i);
	foundDuplicate = true;
      }
      LitFound[getLitsElem(i, j)] = true;
    }
    for(j = 0; j < getCurLen(i); j++)
      LitFound[getLitsElem(i, j)] = false;
  }

  if(foundDuplicate)
    foundBug = true;

#ifdef _DETAILS
  else
    printf("No duplicate variables, implications, or clauses found.\n");
#endif // _DETAILS

}

// This function checks that every implication in the theory has a
// matching contrapositive implication.  ie. if a->b is in the theory,
// it checks that -b->-a is also in the theory.  It also ensures that
// there are no self-implications (a->a) because these are trivially
// true and provide no information.
void checkImplications() {
  int i, j, k, negL1, negL2;
  bool foundMatch, allMatched = true, noselfs = true;

  // Go through all implications looking for their match.
  for(i = 0; i < NLITS; i++) {
    if(active(i)) {
      for(j = 0; j < getNumBcls(i); j++) {
	// Check for self-implications.
	if(i == getLitBclsElem(i, j)) {
	  printf("ERROR: Self implication %d->%d in theory!\n",
		 IntLitToExt(i), IntLitToExt(getLitBclsElem(i, j)));
	  noselfs = false;
	}
	// Check for matching contrapositive implication.
	foundMatch = false;
	negL1 = negate(i);
	negL2 = negate(getLitBclsElem(i, j));
	for(k = 0; k < getNumBcls(negL2); k++) {
	  if(getLitBclsElem(negL2, k) == negL1) {
	    foundMatch = true;
	    break;
	  }
	}
	if(!foundMatch) {
	  printf("ERROR: Implication %d->%d does not have a match!\n",
		 IntLitToExt(i), IntLitToExt(getLitBclsElem(i, j)));
	  allMatched = false;
	}
      }
    }
  }
  if(!(allMatched && noselfs))
    foundBug = true;

#ifdef _DETAILS
  else {
    printf("All implications have a matching contrapositive implication.\n");
    printf("No self-implications founds in the theory.\n");
  }
#endif // _DETAILS

}

// This function checks that certain implications, which should have been
// resolved, do not appear in the theory.  This includes the case where
// a literal 'a' has both the implication a->b and the implication a->-b
// on its LitBcls array.  Also, it is checked that no literal implies
// itself (a->a) or its negation (a->-a).  Finally, nary clauses are
// checked to make sure that a literal and its negation do not both
// appear in the clause. The function returns true if no resolvable
// clauses are found (this is good) and false otherwise.

// NOTE: checkResolvable can only be run right after binary resolution is
// performed or at the end of the preprocessor because in other places the
// code may contain implications that could be resolved.

bool checkResolvable() {
  bool * LitFound = new bool[NLITS];
  bool foundResolvable = false;
  int i, j;
  for(i = 0; i < NLITS; i++)
    LitFound[i] = false;

  // Check for resolvable implications.
  for(i = 0; i < NLITS; i++) {
    if(active(i)) {
      for(j = 0; j < getNumBcls(i); j++) {
	if(getLitBclsElem(i, j) == i) {
	  printf("ERROR: Found self-implication %d->%d!\n",
		 IntLitToExt(i), IntLitToExt(getLitBclsElem(i, j)));
	  foundResolvable = true;
	}
	else if(getLitBclsElem(i, j) == negate(i)) {
	  printf("ERROR: Found lit that should have been forced,  %d->%d!\n",
		 IntLitToExt(i), IntLitToExt(getLitBclsElem(i, j)));
	  foundResolvable = true;
	}
	else if(LitFound[negate(getLitBclsElem(i, j))]) {
	  printf("ERROR: Found both %d->%d and %d->%d in theory!\n",
		 IntLitToExt(i), IntLitToExt(getLitBclsElem(i, j)),
		 IntLitToExt(i), IntLitToExt(negate(getLitBclsElem(i, j))));
	  foundResolvable = true;
	}
	LitFound[getLitBclsElem(i, j)] = true;
      }
      for(j = 0; j < getNumBcls(i); j++)
	LitFound[getLitBclsElem(i, j)] = false;
    }
  }

  // Check for occurances of a literal and its negation in the same nary
  // clause.
  for(i = 0; i < NUMNCLS; i++) {
    for(j = 0; j < getCurLen(i); j++) {
      if(LitFound[negate(getLitsElem(i, j))]) {
	printf("ERROR: Found literals %d and %d in clause %d!\n",
	       IntLitToExt(getLitsElem(i, j)),
	       IntLitToExt(negate(getLitsElem(i, j))), i);
	foundResolvable = true;
      }
      LitFound[getLitsElem(i, j)] = true;
    }
    for(j = 0; j < getCurLen(i); j++)
      LitFound[getLitsElem(i, j)] = false;
  }

  if(foundResolvable)
    foundBug = true;

#ifdef _DETAILS
  else
    printf("Could not find any easily resolvable clauses in the theory.\n");
#endif // _DETAILS

  return(!foundResolvable);
}

// NEW BUCKET
/*
void checkBclsTables() {
  int i, j;

  for(i = 0; i < NLITS; i++) {
    if(active(i)) {
      for(j = 0; j < getNumBcls(i); j++) {
	if(!findBclsTable(i, getLitBclsElem(i, j))) {
	  printf("ERROR: %d->%d not found in hashtable!\n",
		 IntLitToExt(i), IntLitToExt(getLitBclsElem(i, j)));
	  foundBug = true;
	}
      }
    }
  }
}
*/
// END NEW BUCKET
