/*

2clsQ. Copyright Horst Samulowitz and Fahiem Bacchus 2006

 A DPLL QBF solver based on performing extensive reasoning with the binary
 clause subtheory. 

Version 1.0

The software available at this site is copyright (c) 2006 by Horst Samulowitz 
and Fahiem Bacchus. All right are reserved. 
Use of this software is permitted for non-commercial research purposes, and it may be copied only for that use. 
All copies must include this copyright message. This software and
any documentation and/or information supplied with it is distributed
on an as is basis. Horst Samulowitz and Fahiem Bacchus and the University of Toronto make no warranties, express or implied, including but not limited to
implied warranties of merchantability and fitness for a particular
purpose, regarding the documentation, functions or performance of such
software, documentation and/or information.

@misc{Samulowitz:Bacchus:2clsQ,
	author  = { Horst Samulowitz and Fahiem Bacchus },
	year    = { 2006 },
	title   = { {2clsQ} a {DPLL} solver employing extensive binary clause reasoning },
	note    = { Available from
                    http://www.cs.toronto.edu/\~{}fbacchus/sat.html }
}

*/

#ifndef __2clsEq_H
#define __2clsEq_H
#include <vector>
#include <stdlib.h>
#include <stdio.h>

// Horst: Use assert include
#include <assert.h>
// END
//#include <vector.h>

//#pragma inline_depth(255)
//#pragma inline_recursion(on)

#undef NULL
#define NULL 0
//#define _TRACE
//#define _PARANOIA
//#define _DEBUG

using namespace std;

/************************************************************
  globals and class declarations.
************************************************************/

//Number of variables and clauses
extern int NVARS, NCLAUSES, NLITS, NACTIVELITS;
extern bool DOEQS;
extern int LEVEL;

//possible truth values.
enum tval {F, U, T};

class LitCls;
class BinLitCls;
class ClsLit;
class Clause;
class CLAUSELIST;
class Lit;

//Flags
extern bool DPL_debug;
extern bool print_solutions;

//for maintaining reasons.
class DAG;
extern DAG dagTrue;

class Contradiction;

void setContradiction(DAG* d1, DAG* d2);
bool haveContradiction();

// Horst: Add Info about MaxPrefixLevel
extern int NMAXPREFIXLEVEL;


/************************************************************
  Function Prototypes
************************************************************/

#ifdef WIN32
void __cdecl panic(char *fs, ...);
#else
void panic(char *fs, ...);
#endif

void printLit(Lit *l, FILE* fp=stdout);
void printParents(Lit *l, FILE* fp=stdout);
void printNClauses(LitCls *litclslist, FILE* fp=stdout);
void printBClauses(BinLitCls *litBclslist, FILE* fp=stdout);
void printLitCls(LitCls *lc, FILE* fp=stdout);
void printBinLitCls(BinLitCls *lc, FILE* fp=stdout);
void printClause(Clause *c, FILE* fp=stdout);
void printTheory(char* file=NULL);
void printLits(char* file=NULL);
void printCLAUSELIST(CLAUSELIST *cl, FILE* fp=stdout);
void printOriginalClause(Clause *pcls, FILE* fp=stdout);
void printCLAUSELISTANDLEVELS(CLAUSELIST *cl, FILE* fp=stdout);
void printCLAUSEARRAY(int *cl, int len, FILE* fp=stdout);
void checkClauses();
void checkHashTable();
void checkClauseLens(char *s,int l);
void printfalseClauses();
void storeClauseLens(char *s,int l);
void checkLitClauseNums();

bool readProblem(char * fileName);
void initprob();
void initResolver();
void initDPL();
//void initBClsMarkTable();
void initHeur();
void initNoGoods();
void initBitVectorStore(int numlits);

void MakePairofBinLitClsStore();
void UnitProp(int lit);

bool member(int l, int* array, int len);

void initProblemStats();
void recordInitTime();
void recordTime(char *s);

void printProblemStats(char *);
int DPL(int level);
int findNext(int level);
void CheckSolution();
void computeNewClauses(int i);
// Horst: Changed
//void setLiteralScore(int nLiteral, float nScore);
//void setLiteralScorePlus(int nLiteral, float nAddedScore);
//void initLiteralScore(int nNumberofLits);
void createLiteralOrderingByPrefixLevel();
int  findNextPL();
int  findNext2ObeyPL(int);
int  findNext3ObeyPL(int);
int  findNext4ObeyPL(int);
int  findNext5ObeyPL(int);
int  findNext6ObeyPL(int);
int  findNext7ObeyPL(int);
int  findNext8ObeyPL(int);
int findNext7ObeyPL(int, bool);
bool bTheoryEmpty();
int satisfyRemainingClauses();
int findNext8ObeyPL(int);
int findNext9ObeyPL(int);
bool bTheoryEmpty();
int satisfyRemainingClauses();


/***************************************************************
  Some early defnintions of inlines.
***************************************************************/
inline int variable_of(int lit) {
#ifdef PARANOIA
  if (lit < 0 || lit >= 2*NVARS)
    panic("variable_of: passed bad argument %d/%d",lit,NVARS);
#endif
  return(lit/2);
}

inline bool truth_of(int lit) {
#ifdef PARANOIA
  if (lit < 0 || lit >= 2*NVARS)
    panic("truth_of: passed bad argument %d/%d",lit,NVARS);
#endif
  return( (lit & 1) != 0 );
}

//Convert a variable and a sign to file format
inline int to_ff(int var, bool val) {
#ifdef PARANOIA
  if (var < 0 || var >= NVARS)
    panic("to_ff: bad arguments %d/%d,%d",var,NVARS,val?1:0);
#endif
  return( (var + 1) * (val? 1 : -1) );
}

//an interal literal to file format
inline int to_ff(int lit) {
#ifdef PARANOIA
  if (lit < 0 || lit >= 2*NVARS)
    panic("to_ff: passed bad argument %d/%d",lit,NVARS);
#endif
  return( (variable_of(lit) + 1) * (truth_of(lit) ? 1 : -1) );
}

/************************************************************
  Inital Data structures
************************************************************/

//Initial data structures for holding clauses before they are
//transformed by equivalence processing.

class CLAUSELIST {
  //List of clauses.
public:
  int *cls;
  int clslen;
  bool active;
  CLAUSELIST *pnext;
};

//break up the list of clauses into clauses of length N.
const int CLSGROUPS=256;
extern CLAUSELIST *lenNclauses[CLSGROUPS];
//Input limitation---can be altered without negative consequence.
const int MAXCLAUSELEN=50000;


/************************************************************
  Runtime Data structures
************************************************************/

//item in a lit's binary clause list.

class BinLitCls {
public:
  BinLitCls *pnext, *pprev;
  Lit *otherlit;
  BinLitCls *otherlinks;
  DAG *dagnode;
};

//item in a Lit's nary clause list.
class LitCls {
public:
  LitCls *pnext, *pprev; //double link
  Clause *pcls;       //pointer to the clause
  ClsLit *linkincls;  //the particular link in the clause's linked list.
};

//Global array of literals, and Clauses.

extern Lit *THELITS;
extern Clause *THECLAUSES;

//For storing a literal with links to the clauses they participate in.
//The structure contains the number of the literal, the literal it is
//equivalent to, the truth value of the literal. And a list of n-ary
//clauses containing the literal, and binary clauses containing the
//literal. The lists of clauses are doubly linked with head and tail
//markers. The list of binary causes has as its head the literals
//negation.

inline int negate(int l) {
  return(l ^ 1);
}

//Each lit has a hash table to quickly locate the 2clauses it participates in.

const int NOLIT = -1;
const int NOCLS = -1;

//enum Hval {INSTACK, RESOLVED};

typedef struct bucket {
  bucket *pnext;
  int lit;
  //  Hval val;
  DAG *dagnode;
} bucket;

//size of this hash table.
const int TSIZEPWR2 = 8;
const int TSIZE = 1 << TSIZEPWR2;

//each lit also has a bit vector for constant time space
//efficient testing of the presense of 2-clauses.

typedef unsigned int bit;
const int bitsperint=sizeof(int)*8;
inline int numberOfInts(int n) {
  //number of ints required to store n bits.
  return((n+bitsperint-1)/bitsperint);
}
extern bit *bitvectorStore;

inline void setbit(bit *vector, int i) {
  vector[i/bitsperint] |= (1 << (i%bitsperint));
}

inline void unsetbit(bit *vector, int i) {
  vector[i/bitsperint] &= ~(1 << (i%bitsperint));
}

inline bool getbit(bit *vector, int i) {
  if(vector[i/bitsperint] & (1 << (i%bitsperint)))
	return(true);
  else
	return(false);
}

class Lit {
public:
  int num;      //the number of this literal
  Lit* equivlit; //the literal this literal is equivalent to.
  tval val;     //the literals truth value.
  DAG *dagnode;
  DAG *dagEqnode;
  LitCls nClauses;     //a list of n-ary clauses literal appears in
  LitCls nendMark;
  //a list of binary clauses literal appears in.

  // Horst: ADDED PREXFIXTYPE AND PREFIXLEVE
  int  nPrefixLevel;
  bool bExistential;
  long lCube1;
  long lCube2;
  vector<int> vecEquivalences;
  // Change END

  //Always contains the negation of the literal itself.
  BinLitCls twoClauses;
  BinLitCls twoendMark;
  int numBcls;
  int numNcls;
  int num3cls;
  int assignLevel;
  bucket *litHtable;
  bit *bitvector;
  void initLit(int i) {
	int j;
    num=i;
    equivlit=THELITS+i;
    val=U;
    dagnode=NULL;
	dagEqnode=&dagTrue;
    //set up Nary clause list
    nClauses.pnext=&nendMark;
    nClauses.pprev=NULL;
    nendMark.pnext=NULL;
    nendMark.pprev=&nClauses;
    nClauses.pcls=nendMark.pcls=NULL;
    nClauses.linkincls=nendMark.linkincls=NULL;
    //set up binary clause list
    twoClauses.pnext=&twoendMark;
    twoClauses.pprev=NULL;
    twoClauses.otherlit=THELITS+negate(i);
    twoClauses.otherlinks=&((THELITS+negate(i))->twoClauses);
    twoClauses.dagnode = &dagTrue;
    twoendMark.pnext=NULL;
    twoendMark.pprev=&twoClauses;
    twoendMark.otherlit=NULL;
    twoendMark.otherlinks=NULL;
    numBcls=1;
    numNcls=0;
    num3cls=0;
    assignLevel=0;
	lCube1 = 0;
    lCube2 = 0;
	vecEquivalences.clear();


    litHtable=new bucket[TSIZE];
    if(!litHtable)
      panic("Memory Allocation Failure #1\n");

    for(j=0;j<TSIZE;j++) {
      litHtable[j].lit = NOLIT;
      litHtable[j].pnext = NULL;
    }
	int size=numberOfInts(i+1);
	bitvector=bitvectorStore;
	bitvectorStore +=size;
	for(j=0;j<size;j++)
	  bitvector[j]=0;
  };
};

inline Lit* intTolit(int l) {
  return(THELITS+l);
}

inline Lit* negate(Lit *pl) {
  return(pl->twoClauses.otherlit);
}

inline Lit *eqlit(Lit *L) {
  while(L->equivlit!=L)
    L=L->equivlit;
  return(L);
}

inline int eqlit(int l) {
  return(eqlit(intTolit(l))->num);
}

inline tval val(Lit* l) {
  return(l->val);
}

inline tval val(int l) {
  return(val(intTolit(l)));
}


inline int litToint(Lit *l) {
  return(l->num);
}

inline bool isInactive(Lit *l) {
  return(l->equivlit!=l || l->val!=U);
}

inline bool isInactive(int l) {
  return(isInactive(intTolit(l)));
}

inline bool isActive(Lit *l) {
  return(l->equivlit==l && l->val==U);
}

inline bool isActive(int l) {
  return(isActive(intTolit(l)));
}

inline int numBcls(Lit *l) {
  return(l->numBcls);
}

inline int numBcls(int l) {
  return(numBcls(intTolit(l)));
}

inline int numNcls(Lit *l) {
  return(l->numNcls);
}

inline int numNcls(int l) {
  return(numNcls(intTolit(l)));
}

inline int num3cls(Lit *l) {
  return(l->num3cls);
}

inline int num3cls(int l) {
  return(num3cls(intTolit(l)));
}


inline int assignLevel(Lit *l) {
  return(l->assignLevel);
}

inline int assignLevel(int l) {
  return(assignLevel(intTolit(l)));
}

inline int to_ff(Lit *l) {
  return(to_ff(l->num));
}

inline void setLitTrue(Lit* L) {
#ifdef _TRACE
  printf("=>setLitTrue(%d)<=\n",to_ff(L->num));
#endif _TRACE
//#ifdef _PARANOIA
  if(L->val==F)
    printf("=>setLitTrue(%d), lit is already false!\n", to_ff(L->num));
  if(L->val==T)
    printf("=>setLitTrue(%d), lit is already true!\n", to_ff(L->num));
  //printf("\n Setting %d (Internal: %d) at level: %d ", to_ff(L->num), L->num, LEVEL);
//#endif _PARANOA
  //printf("\n%d 0", to_ff(L->num));
  //if((L->num == 385) || (L->num == 331) || (L->num == 329)) 
  // printf("\n TTTSetting %d (Internal: %d) at level: %d (E?: %d)", to_ff(L->num), L->num , LEVEL, L->bExistential);
  L->val=T;
  L->twoClauses.otherlit->val=F;
  L->assignLevel=LEVEL;
  L->twoClauses.otherlit->assignLevel=LEVEL;

}

inline void setLitTrue(int l) {
  setLitTrue(intTolit(l));
}

inline void setLitUnassigned(Lit* L) {
#ifdef _TRACE
  printf("=>setLitUnassigned(%d)<=\n",to_ff(L->num));
#endif _TRACE
  L->val=U;
  L->twoClauses.otherlit->val=U;
  L->assignLevel=0;
  L->twoClauses.otherlit->assignLevel=0;
  L->lCube1 = 0;
  L->lCube2 = 0;
  //if((L->num== 100) || (L->num==101))
	 // printf("\n YYYYY Undoing %d! ", L->num);
}

inline void setLitdagnode(Lit* L, DAG* dagnode) {
  L->dagnode=dagnode;
}

inline void setLitdagnode(int l, DAG* dagnode) {
  setLitdagnode(intTolit(l),dagnode);
}

inline DAG *getLitdagnode(Lit* L) {
  return(L->dagnode);
}

inline DAG *getLitdagnode(int l) {
  return(getLitdagnode(intTolit(l)));
}

inline void setLitEqdagnode(Lit* L, DAG* dagnode) {
  L->dagEqnode=dagnode;
}

inline void setLitEqdagnode(int l, DAG* dagnode) {
  setLitEqdagnode(intTolit(l),dagnode);
}

inline DAG *getLitEqdagnode(Lit* L) {
  return(L->dagEqnode);
}

inline DAG *getLitEqdagnode(int l) {
  return(getLitEqdagnode(intTolit(l)));
}

inline bool isTrue(Lit *l) {
  //is literal number l true?
  return(l->val==T);
}

inline bool isTrue(int l) {
  //is literal number l true?
  return(isTrue(intTolit(l)));
}

inline bool isValued(Lit *l) {
  //is literal number l true?
  return(l->val!=U);
}

inline bool isValued(int l) {
  //is literal number l true?
  return(isValued(intTolit(l)));
}


inline bool isUnvalued(Lit *l) {
  //is literal number l true?
  return(l->val==U);
}

inline bool isUnvalued(int l) {
  //is literal number l true?
  return(isUnvalued(intTolit(l)));
}


inline bool isFalse(Lit *l) {
  //is literal number l true?
  return(l->val==F);
}

inline bool isFalse(int l) {
  //is literal number l true?
  return(isFalse(intTolit(l)));
}

// *********************************************************
// Horst: ADDED Prefix-Operators
// *********************************************************

inline void setLitPrefixLevel(Lit * L, int nLevel) {
  L->nPrefixLevel = nLevel;
}

inline void setLitPrefixLevel(int l, int nLevel) {
  intTolit(l)->nPrefixLevel = nLevel;
}

inline int getLitPrefixLevel(Lit* L) {
  return L->nPrefixLevel;
}

inline int getLitPrefixLevel(int l) {
  return (intTolit(l)->nPrefixLevel);
}

inline void setLitPrefixType(Lit* L, bool bExistential) {
  L->bExistential = bExistential;
}

inline void setLitCube1(Lit* L, long lCube) {
  L->lCube1 = lCube;
}

inline void setLitCube1(int l, long lCube) {
  intTolit(l)->lCube1 = lCube;
}

inline void setLitCube2(Lit* L, long lCube) {
  L->lCube2 = lCube;
}

inline void setLitCube2(int l, long lCube) {
  intTolit(l)->lCube2 = lCube;
}

inline void setLitPrefixType(int l, bool bExistential) {
  intTolit(l)->bExistential = bExistential;
}

inline bool isLitExistential(Lit* L) {
  return L->bExistential;
}

inline bool isLitExistential(int l) {
  return (intTolit(l)->bExistential);
}

inline int getMaxLiteral()
{
  return NLITS;
}

inline int getReasonDecisionLevel(int l)
{
  Lit* L = intTolit(l);
  return L->assignLevel;
}

inline int getReasonDecisionLevel(Lit * L)
{
  return L->assignLevel;
}


/*// Jessica
inline int getLowestEquiv(preLit* L) {
  return L->lowestEquiv;
}

inline int getLowestEquiv(int l) {
  return (PRELITS[l].lowestEquiv);
}
inline void setLowestEquiv(preLit* L, int u) {
  L->lowestEquiv = u;
}

inline void setLowestEquiv(int l, int u) {
  PRELITS[l].lowestEquiv = u;
}
*/
// *********************************************************
// END: ADDED Prefix-Operators
// *********************************************************




//item in a n-aryclauses lit list.
class ClsLit {
public:
  ClsLit *pnext, *pprev;
  Lit *plit;
  LitCls *linkinlit; //the particular link in the lit's linked list of clauses.
};

//for storing a nonbinary clause
class Clause {
  //The list of literals in a clause is stored as a doubly linked
  //list with a head and tail marker.
public:
  int num;       //the number of this clause
  int len;       //number of literals in this clause
  bool active;
  ClsLit start;  //a of list literals in this clause
  ClsLit end;  //a of list literals in this clause
  CLAUSELIST *originalClause;
  DAG *dagnode;
  void initClause(int i) {
    num=i;
    len=0;
    start.pnext=&end;
    start.pprev=NULL;
    end.pnext=NULL;
    end.pprev=&start;
    start.plit=end.plit=NULL;
    start.linkinlit=end.linkinlit=NULL;
    active=true;
    dagnode=&dagTrue;
  };
};

inline Clause* intToclause(int c) {
  return(THECLAUSES+c);
}

inline int clauseToint(Clause *c) {
  return(c->num);
}

inline bool clsisActive(Clause *c) {
  return(c->active);
}

inline bool clsisActive(int c) {
  return(clsisActive(intToclause(c)));
}

class PairofBinLitCls {
public:
  BinLitCls Lit1;
  BinLitCls Lit2;
  PairofBinLitCls() {
    Lit1.pnext=Lit1.pprev=Lit2.pnext=Lit2.pprev=NULL;
    Lit1.otherlit=Lit2.otherlit=NULL;
    Lit1.otherlinks = &Lit2;
    Lit2.otherlinks = &Lit1;
  };
};


/***********************************************************************
          VARIABLES, CLAUSES, LITERALS
***********************************************************************/

/*Because it's C, its most convenient to have clauses, variables, and
  literals be 0 based.
  Thus:
  Clauses are 0 ... number_of_clauses -1

  Variables are 0 ... number_of_variables - 1

  Truth values are booleans (i.e. type bool)

  Literals range from 0 ... 2*number_of_variables -1
  The interpretation is:
  variable i, false <---> 2 * i
  variable i, true  <---> 2 * i + 1

  However, the file format represents things differently; there
  variables are 1-based, and literals use -ve or +ve to indicate sign.
  The program converts from file format <--> internal format when
  reading and writing files (and note also, clauses are 1 based in
  file format as well).  Most on-screen messages use file format as
  well. But aside from these places, everything is always internal
  format. Whether we work with literals  or with decomposed
  variables+truth value depends on context; there are some simple
  pieces of code below to do the conversion.
*/

inline int to_literal(int var, bool val) {
#ifdef PARANOIA
  if (var < 0 || var >= NVARS)
    panic("to_literal: bad arguments %d/%d,%d",var,NVARS,val?1:0);
#endif
  return(2*var+(val?1:0));
}

//a literal from file format to internal.
inline int to_internal(int lit) {
#ifdef PARANOIA
  if (abs(lit) < 1 || abs(lit) > NVARS)
    panic("to_internal: bad argument %d",lit);
#endif
  return( to_literal(abs(lit)-1, lit<0?0:1) );
}

//an interal literal to file format
inline char ff_tval(tval val) {
  //return a character indicating the truth value.
  if(val==F) return('F');
  else if(val==T) return('T');
  else return('U');
}

// Horst: get MaxPrefixLevel
inline int getMaxPrefixLevel() {
 return NMAXPREFIXLEVEL;
}


struct stNoGoodInformation {
	int nForcedLit;
	DAG* dagnode;
};

struct stEqInformation {
	Lit* lLit;
	int  nEquivalences;
};

#endif /* __2clsEq_H */
