/* slaney.c -- based on John Slaney's Random Blocks World Generator

This routine generates random states for Blocks World (BW).  The states
are uniformly distributed over the class of all possible states for the
given number of blocks. 

The blocks are represented internally as integers between 0 and n-1, while
the table is represented as -1.  For example, the state of 6 blocks

	1
	4	2
	3	0	5
	---------

is represented internally by the integers

	-1, 4, 0, -1 3 -1

meaning 
	0 is on the table, 
	1 is on 4, 
	2 is on 0, 
	3 is on the table, 
	4 is on 3 and 
	5 is on the table.

Sample runs, 7 blocks, seed 3088 (these are origin 0 instead of -1):  

4 3 0 5 7 1 0
2 0 7 1 6 4 0
2 5 6 7 3 0 0
0 0 6 2 0 0 5

The random numbers are generated by the function drand48(). To
ensure that every BW state of the given size has the same probability of
being generated, it is necessary to calculate the probability at every
step that the next block to be placed should go on the table. The numbers
required for this are calculated in advance.

Memory requirements are quadratic in nBlocks.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#ifdef WIN32
#include <windows.h>
#endif /* WIN32 */

#include "../tlplan.h"
#include "../eval.h"
#include "../formula.h"
#include "../list.h"
#include "../makeform.h"
#include "../tl_tab.h"
#include "../util.h"

typedef struct							/* A tower is distinguished by its top and bottom blocks */
{
	int nTop;
	int nBottom;
}TOWER, *TOWERP;

/* local function prototypes */

static CELLP MakeLiterals
(
	BOOL bHandEmpty,					// handempty flag
	int nBlocks,						// number of blocks
	int *pnState						// block states
);
static CELLP MakeGoalLiterals
(
	BOOL bHandEmpty,					// handempty flag
	int nBlocks,						// number of blocks
	int *pnState						// block states
);
static CELLP MakeClearLiterals
(
	BOOL bTable,						/* generate (clear table) literal */
	int nBlocks,						// number of blocks
	int *pnState						// block states
);
static CELLP MakeOnTableLiterals
(
	int nBlocks,						// number of blocks
	int *pnState						// block states
);
static CELLP MakeOnLiterals
(
	int nBlocks,						// number of blocks
	int *pnState						// block states
);
static int *BWState
(
	int nBlocks							/* number of blocks */
);
static double Ratio
(
	double *pdfRatio,
	int n,
	int i,
	int j
);
static int Pos(int n, int x, int y);
static void MakeRatio(int n, double *pdfRatio);
static int nint
(
	double x
);
static CELLP BlockName
(
	int nIndex							/* block index */
);

static double drand48(void);
static void dorand48
(
	unsigned short xseed[3]
);
static void srand48
(
	long seed
);
static unsigned short *seed48
(
	unsigned short xseed[3]
);

/* Slaney Block Problem Generators ---------------------------------------------

Functions for generating a random blocks world initial state and goal state.

You can execute (set-initial-world (slaney-blocks-world <seed> 25))
to get an random initial world configuration with 25 blocks.

You can also use (set-goal (slaney-blocks-world <seed> 25))
to set up a random goal with 25 blocks.

(slaney-blocks-goal ...) is provided as usually you want to use incompletely
specified goals.

*/

/* SlaneyBlocksWorld (Function)

Description:
	Generate a list of literals describing a complete configuration of
	an nLimit blocks world.
*/

DECLSPECX CELLP SlaneyBlocksWorld
(
	CELLP pcFormula,
	LINEARPLANP plpLinearPlan,
	BINDINGP pbBindings
)
{
	char acBuffer[64];
	CELLP pc;
	int nBlocks;
	int n;
	int *pnState;

	/* get the plan number (seed) */
	
	pc=pcFormula->pfForm->pcArgs;
	if(!FormulaToInteger(pc,plpLinearPlan,pbBindings,&n))
		return NULL;					/* message already printed */
	if(n!=-1)							/* default the seed if n==-1 */
		srand48(n);						/* initialize drand48() */

	/* get the number of blocks */
	
	pc=pc->pcNext;
	if(!FormulaToInteger(pc,plpLinearPlan,pbBindings,&nBlocks))
		return NULL;					/* message already printed */
	if(nBlocks<1||nBlocks>=12356630)
	{
		ErrorMessage("slaney-blocks-world:  Number of blocks must between 1 and %d\n",12356630-1);
		return NULL;
	}

	/* fill in the plan name */
	
	if(n!=-1)
	{
		sprintf(acBuffer,"%d Blocks, Plan %d",nBlocks,n);
		psPlanName=StrAlloc(acBuffer);
	}

	/* generate the literals */
	
	pnState=BWState(nBlocks);
	pc=MakeLiterals(TRUE,nBlocks,pnState);
	free(pnState);
	return pc;
}

/* MakeLiterals (Local Function)

Description:
	Generate a list of literals representing the state of the blocks.
*/

static CELLP MakeLiterals
(
	BOOL bHandEmpty,					// handempty flag
	int nBlocks,						// number of blocks
	int *pnState						// block states
)
{
	CELLP pcStart,pcEnd;

	pcStart=NULL;
	pcEnd=(CELLP)&pcStart;

	pcEnd->pcNext=MakeClearLiterals(FALSE,nBlocks,pnState);
	while(pcEnd->pcNext)
		pcEnd=pcEnd->pcNext;

	pcEnd->pcNext=MakeOnTableLiterals(nBlocks,pnState);
	while(pcEnd->pcNext)
		pcEnd=pcEnd->pcNext;

	pcEnd->pcNext=MakeOnLiterals(nBlocks,pnState);
	while(pcEnd->pcNext)
		pcEnd=pcEnd->pcNext;

	if(bHandEmpty)
		pcEnd->pcNext=MakeSymbolInfoForm(FALSE,NewList(ATOM_IDENT,"handempty"),NULL);
	return pcStart;
}

/* SlaneyBlocksGoal (Function)

Description:
	Generate a list of literals describing an incomplete configuration of
	an nCount blocks world.  In particular do not include any ontable or clear
	literals.
*/

DECLSPECX CELLP SlaneyBlocksGoal
(
	CELLP pcFormula,
	LINEARPLANP plpLinearPlan,
	BINDINGP pbBindings
)
{
	char acBuffer[64];
	CELLP pc;
	int nBlocks;
	int n;
	int *pnState;						/* block state vector */

	/* get the plan number */
	
	pc=pcFormula->pfForm->pcArgs;
	if(!FormulaToInteger(pc,plpLinearPlan,pbBindings,&n))
		return NULL;					/* message already printed */
	if(n!=-1)							/* default the seed if n==-1 */
		srand48(n);						/* initialize drand48() */

	/* get the number of blocks */
	
	pc=pc->pcNext;
	if(!FormulaToInteger(pc,plpLinearPlan,pbBindings,&nBlocks))
		return NULL;					/* message already printed */
	if(nBlocks<1||nBlocks>=12356630)
	{
		ErrorMessage("random-blocks-goal:  Block limit must between 1 and %d\n",12356630-1);
		return NULL;
	}

	/* fill in the plan name */
	
	if(n!=-1)
	{
		sprintf(acBuffer,"%d Blocks, Plan %d",nBlocks,n);
		psPlanName=StrAlloc(acBuffer);
	}

	/* generate the literals */
	
	pnState=BWState(nBlocks);
	pc=MakeLiterals(TRUE,nBlocks,pnState);
	free(pnState);
	return pc;
}

/* MakeGoalLiterals (Local Function)

Description:
	Generate a list of literals representing the state of the blocks.
Notes:
	We do not generate any (clear C) or (ontable c) predicates.
*/

static CELLP MakeGoalLiterals
(
	BOOL bHandEmpty,					// handempty flag
	int nBlocks,						// number of blocks
	int *pnState						// block states
)
{
	CELLP pcStart,pcEnd;

	pcStart=NULL;
	pcEnd=(CELLP)&pcStart;

	pcEnd->pcNext=MakeOnTableLiterals(nBlocks,pnState);
	while(pcEnd->pcNext)
		pcEnd=pcEnd->pcNext;

	pcEnd->pcNext=MakeOnLiterals(nBlocks,pnState);
	while(pcEnd->pcNext)
		pcEnd=pcEnd->pcNext;

	if(bHandEmpty)
		pcEnd->pcNext=MakeSymbolInfoForm(FALSE,NewList(ATOM_IDENT,"handempty"),NULL);
	return pcStart;
}

/* MakeClearLiterals 

Description:
	Generate (clear C) literals.
*/

static CELLP MakeClearLiterals
(
	BOOL bTable,						/* generate (clear table) literal */
	int nBlocks,						// number of blocks
	int *pnState						// block states
)
{
	int i;
	CELLP pcStart,pcEnd;
	CELLP pc;
	LITVAL lv;
	int *pnTemp;

	pcStart=NULL;
	pcEnd=(CELLP)&pcStart;

	// find all the clear blocks in the block state
	
	pnTemp=(int *)malloc(sizeof(int)*nBlocks);
	for(i=0;i<nBlocks;i++)
		pnTemp[i]=-1;					// assume all blocks clear
	for(i=0;i<nBlocks;i++)
	{
		if(pnState[i]>=0)
			pnTemp[pnState[i]]=0;		// this block is covered
	}

	// generate the clear predicates	

	for(i=0;i<nBlocks;i++)
	{
		if(pnTemp[i]<0)
		{
			pc=BlockName(i);
			pcEnd=pcEnd->pcNext=MakeSymbolInfoForm(FALSE,NewList(ATOM_IDENT,"clear"),pc);
		}
	}
	free(pnTemp);

	if(bTable)
	{
		lv.psString="table";
		pc=MakeLiteralForm(ATOM_IDENT,lv);
		pcEnd=pcEnd->pcNext=MakeSymbolInfoForm(FALSE,NewList(ATOM_IDENT,"clear"),pc);
	}

	return pcStart;
}

/* MakeOnTableLiterals 

Description:
	Generate (ontable C) literals.
*/

static CELLP MakeOnTableLiterals
(
	int nBlocks,						// number of blocks
	int *pnState						// block states
)
{
	int i;

	CELLP pcStart,pcEnd;
	CELLP pc;

	pcStart=NULL;
	pcEnd=(CELLP)&pcStart;

	/* generate ontable predicates */

	for(i=0;i<nBlocks;i++)
	{
		if(pnState[i]<0)
		{
			pc=BlockName(i);
			pcEnd=pcEnd->pcNext=MakeSymbolInfoForm(FALSE,NewList(ATOM_IDENT,"ontable"),pc);
		}
	}
	return pcStart;
}

/* MakeOnLiterals 

Description:
	Generate (on A C) literals.
*/

static CELLP MakeOnLiterals
(
	int nBlocks,						// number of blocks
	int *pnState						// block states
)
{
	int i;
	CELLP pcStart,pcEnd;
	CELLP pc1,pc2;

	pcStart=NULL;
	pcEnd=(CELLP)&pcStart;

	/* generate on predicates from each pair of blocks in each stack */

	for(i=0;i<nBlocks;i++)
	{
		if(pnState[i]>=0)
		{
			pc1=BlockName(i);
			pc2=BlockName(pnState[i]);
			pc1->pcNext=pc2;
			pcEnd=pcEnd->pcNext=MakeSymbolInfoForm(FALSE,NewList(ATOM_IDENT,"on"),pc1);
		}
	}
	return pcStart;
}

/* BWState ---------------------------------------------------------------------

Description:
	Generate towers using Slaney's algorithm.
  	To make the state, begin by regarding the blocks as short Floating towers,
	and repeatedly take the last one and put it on something. It may go on
	the table, in which case the array of grounded or rooted towers is
	extended by one, or it may go on another (Floating or rooted) tower. All
	destinations except for the table have equal probability.
Note:
	The returned integer vector should be free'd when it is no longer needed.
*/

static int *BWState
(
	int nBlocks							/* number of blocks */
)
{
	int i;
	double r;							/* The randomly generated number */
	double rat;							/* The relevant ratio from the array */
	double p;							/* The probability that the block goes on the table */
	int choice;							/* Abbreviates (n + k) */
	int b;								/* The destination block */

	int nSize;							/* ratio array size */
	double *pdfRatio;					/* ratio array */
	int *pnOn;							/* The ON relation considered as a function */
	TOWER *ptRooted;					/* The towers which are on the table */
	TOWER *ptFloating;					/* The towers which are not yet on anything */
	int nrt;							/* The number of rooted towers */
	int nft;							/* The number of Floating towers */

	// allocate data structures
	
	nSize=((nBlocks+2)*(nBlocks+2)+3)/4;
	pdfRatio=(double *)malloc(sizeof(double)*nSize);
	pnOn=(int *)malloc(sizeof(int)*(nBlocks+1));
	ptRooted=(TOWERP)malloc(sizeof(TOWER)*(nBlocks+1));
	ptFloating=(TOWERP)malloc(sizeof(TOWER)*(nBlocks+1));
	nBlocks=nBlocks;
	MakeRatio(nBlocks,pdfRatio);		/* generate probabilities */

	// initialize data
	
	for(i=0;i<nBlocks;i++)
	{
		ptRooted[i].nTop=ptRooted[i].nBottom=-1;
		ptFloating[i].nTop=ptFloating[i].nBottom=i;
		pnOn[i]=-1;
	}
	
	// fill in state
	
	nrt=0;								/* Initially, each block is a Floating tower */
	nft=nBlocks;
	while(nft--) 
	{
		r=drand48();
		choice=nft+nrt;
		rat=Ratio(pdfRatio,nBlocks,nft,nrt);
		p=rat/(rat+choice);
		if(r<=p)						/* Put the next block on the table */
		{
			ptRooted[nrt].nTop=ptFloating[nft].nTop;
			ptRooted[nrt].nBottom=ptFloating[nft].nBottom;
			nrt++;
		}
		else							/* Put the next block on some b */
		{							
			b=nint(floor((r-p)/((1.0-p)/choice)));
			if(b<nrt)					/* Destination is a rooted tower */
			{
				pnOn[ptFloating[nft].nBottom]=ptRooted[b].nTop;
				ptRooted[b].nTop=ptFloating[nft].nTop;
			}
			else
			{							/* Destination is a Floating tower */
				b-=nrt;
				pnOn[ptFloating[nft].nBottom]=ptFloating[b].nTop;
				ptFloating[b].nTop=ptFloating[nft].nTop;
			}
		}
	}
	free(pdfRatio);
	free(ptRooted);
	free(ptFloating);
	return pnOn;
}

/* MakeRatio

Description:
	The function g is easily defined recursively:
		g(0,k) = 1
		g(n+1,k) = g(n,k)(n + k) + g(n,k+1)

	This determines the required ratio. 
	
	Let g(n-1,k) = a. Let g(n-1,k+1)/a = R. Let g(n-1,k+2)/g(n-1,k+1) = S.
	Then we have:
		g(n,k+1) / g(n,k)
			= (Ra(n+k) + SRa) / (a(n+k-1) + Ra)
			= R(n+k+Sa) / (n+k-1+R)
			= (n+k+Sa) / ((n+k-1)/R + 1)

	Either of the last two expressions may be used conveniently to calculate
	the ratio for (n,k) given those for (n-1,k) and (n-1,k+1).
*/

static void MakeRatio
(
	int n, 
	double *pdfRatio
)
{
	int j,k;
	double *pdfTemp;

	pdfTemp=(double *)malloc(sizeof(double)*(n+1));
	for(k=0;k<=n;k++)
	{
		pdfTemp[k]=1.0;
		pdfRatio[Pos(n,0,k)]=1.0;
	}
	for(j=1;j<=n;j++)
	{
		for(k=0;k+j<=n;k++)
		{
			pdfTemp[k]=(pdfTemp[k]*(pdfTemp[k+1]+j+k))/(pdfTemp[k]+j+k-1.0);
			if(!(j%2))
				pdfRatio[Pos(n,j/2,k)]=pdfTemp[k];
		}
	}
	free(pdfTemp);
}

/* Pos

Description:
	The 2-dimensional array of ratios is represented in one dimension, so here
	is an index function such that ratio[pos(x,y)] is essentially ratio[2x][y].
*/

static int Pos
(
	int n,
	int i, 
	int j
)
{
	return (i*(n+2-i))+j;
}

/* Ratio

Description:
	Let g(n,k) be the number of states that extend a part-state with k
	towers already on the table and n Floating towers not yet on anything.
	We work with Ratio(...n,k) which is g(n,k+1)/g(n,k).

	The ratio is stored in the case of an even-numbered row, and calculated
	in the case of an odd-numbered row. 
	This is simply to halve the space required to store ratios. 
	Note that N is the number of blocks.
*/

static double Ratio
(
	double *pdfRatio,
	int n,
	int i,
	int j
)
{
	int k;

	k=Pos(n,i/2,j);
	if(i%2)
		return (pdfRatio[k+1]+i+j)/(((1.0/pdfRatio[k])*(i+j-1))+1.0);
	else 
		return pdfRatio[k];
}

static int nint
(
	double x
)
{
	if(x<0.0)
		return (int)(x-0.5);
	return (int)(x+0.5);
}

/* BlockName -------------------------------------------------------------------

Description:
	Convert a block index into a block name.
*/

static CELLP BlockName
(
	int nIndex							/* block index */
)
{
	LITVAL lv;
	char ac[8];							/* block name buffer */

	if(nIndex<26)						/* 26^1 */
		sprintf(ac,"%c",
			'a'+nIndex);
	else if(nIndex<702)					/* 26^1+26^2 */
		sprintf(ac,"%c%c",
			'a'+(nIndex-26)/26,
			'a'+((nIndex-26)%26));
	else if(nIndex<18278)				/* 26^1+26^2+26^3 */
		sprintf(ac,"%c%c%c",
			'a'+(nIndex-702)/(26*26),
			'a'+((nIndex-702)/26%26),
			'a'+((nIndex-702)%26));
	else if(nIndex<475254)				/* 26^1+26^2+26^3+26^4 */
		sprintf(ac,"%c%c%c%c",
			'a'+(nIndex-18278)/(26*26*26),
			'a'+((nIndex-18278)/(26*26)%26),
			'a'+((nIndex-18278)/26%26),
			'a'+((nIndex-18278)%26));
	else if(nIndex<12356630)			/* 26^1+26^2+26^3+26^4+26^5 */
		sprintf(ac,"%c%c%c%c%c",
			'a'+(nIndex-475254)/(26*26*26*26),
			'a'+((nIndex-475254)/(26*26*26)%26),
			'a'+((nIndex-475254)/(26*26)%26),
			'a'+((nIndex-475254)/26%26),
			'a'+((nIndex-475254)%26));

	lv.psString=ac;
	return MakeLiteralForm(ATOM_IDENT,lv);
}

/* drand48 ---------------------------------------------------------------------

Description:
	Implementation of unix drand48 routines.
*/

#define SEED0	0x330E
#define SEED1	0xABCD
#define SEED2	0x1234
#define MULT0	0xE66D
#define MULT1	0xDEEC
//#define MULT2	0x000F
#define MULT2	0x0005
#define ADD		0xB

static unsigned short seed[3]=
{
	SEED0,
	SEED1,
	SEED2
};
static unsigned short mult[3]=
{
	MULT0,
	MULT1,
	MULT2
};
static unsigned short add=ADD;

/* drand48 */

static double drand48(void)
{
	dorand48(seed);
	return 
		ldexp((double)seed[0],-48)+
		ldexp((double)seed[1],-32)+
		ldexp((double)seed[2],-16);
}

static void dorand48
(
	unsigned short xseed[3]
)
{
	unsigned long accu;
	unsigned short temp[2];

	accu=(unsigned long)mult[0]*(unsigned long)xseed[0]+
		(unsigned long)add;
	temp[0]=(unsigned short)accu;		/* lower 16 bits */
	accu>>=sizeof(unsigned short)*8;
	accu+=(unsigned long)mult[0]*(unsigned long)xseed[1]+
		(unsigned long)mult[1]*(unsigned long)xseed[0];
	temp[1]=(unsigned short)accu;		/* middle 16 bits */
	accu>>=sizeof(unsigned short)*8;
	accu+=mult[0]*xseed[2]+mult[1]*xseed[1]+mult[2]*xseed[0];
	xseed[0]=temp[0];
	xseed[1]=temp[1];
	xseed[2]=(unsigned short)accu;
}

/* seed48 

Description:
	Seed drand48 with a 48 bit seed.
Returns:
	The previous seed.
*/

static unsigned short *seed48
(
	unsigned short xseed[3]
)
{
	static unsigned short sseed[3];

	sseed[0]=seed[0];
	sseed[1]=seed[1];
	sseed[2]=seed[2];
	seed[0]=xseed[0];
	seed[1]=xseed[1];
	seed[2]=xseed[2];
	return sseed;
}

/* srand48

Description:
	Seed drand48 with a 32 bit seed.
*/

static void srand48
(
	long s
)
{
	seed[0]=SEED0;
	seed[1]=(unsigned short)s;
	seed[2]=(unsigned short)(s>>16);
}
