% [prob,t_mean,t_cov,allX] = mix_bernoulli_distrib(P,Pi)
%
% Given a mixture of D-variate Bernoulli distributions with parameters
% P and Pi, mix_bernoulli_distrib computes the probability
% distribution in D-dimensional binary space.
%
% In:
%   P: MxD matrix containing the Bernoulli probabilities.
%   Pi: Mx1 vector containing the mixing parameters.
% Out:
%   prob: 2^D-dimensional vector containing the probabilities for each 
%      binary vector under the mixture. Note that it is indexed from 1 
%      to 2^D, not from 0 to 2^D-1.
%   t_mean: 1xD vector containing the theoretical mean of the mixture.
%   t_cov: DxD matrix containing the theoretical covariance matrix of
%      the mixture.
%   allX: 2^DxD matrix containing all 2^D binary vectors: row 1
%      contains 0...0 (=0),..., row 2^D contains 1...1 (=2^D-1).
% See also mix_bernoulli, mix_bernoulli_sample, mix_bernoulli_llh_error.

% Copyright (c) 1998 by Miguel A. Carreira-Perpinan

function [prob,t_mean,t_cov,allX] = mix_bernoulli_distrib(P,Pi)

[M,D] = size(P);

% allX: matrix containing all binary numbers of up to D bits
allX = zeros(2^D,D);
for i=0:2^D-1
  s=dec2bin(i);
  % Insert leading zeroes
  for j=1:D-length(s)
    s=['0' s];
  end
  for j=1:D
    allX(i+1,j) = str2num(s(j));
  end
end

[llh,e,Xrec,B,R] = mix_bernoulli_llh_error(allX,P,Pi);
prob=B*Pi;

% Theoretical mean vector and covariance matrix
if nargout>1         % Compute theoretical mean vector
  t_mean = Pi'*P;
end

if nargout>2         % Compute theoretical covariance matrix
  t_cov = diag(Pi'*(P-P.^2)) + P'*(diag(Pi)-Pi*Pi')*P;
end

