% [X,s_mean,t_mean,s_cov,t_cov] = mix_bernoulli_sample(N,P[,Pi])
% Sample from mixture of multivariate Bernoulli distributions
%
% Produces N samples from a mixture of M D-variate Bernoulli
% distributions given by mixture proportions Pi(1..M) and Bernoulli
% parameters P(1..M,:). It also gives the sample and theoretical
% values of the mean and covariance matrix.
%
% If Pi is omitted, the mixture is assumed to be trivial, i.e. a
% D-variate Bernoulli distribution.
%
% In:
%   N: number of samples to generate.
%   P: MxD matrix containing the Bernoulli probabilities.
%   Pi: Mx1 vector containing the mixing parameters (default: 1).
% Out:
%   X: NxD binary data matrix containing the samples rowwise.
%   s_mean: 1xD vector containing the sample mean of the sample generated.
%   t_mean: 1xD vector containing the theoretical mean of the mixture.
%   s_cov: DxD matrix containing the sample covariance matrix of the
%      sample generated.
%   t_cov: DxD matrix containing the theoretical covariance matrix of
%      the mixture.
%
% See also mix_bernoulli, mix_bernoulli_llh_error, mix_bernoulli_distrib.

% Copyright (c) 1997 by Miguel A. Carreira-Perpinan

function [X,s_mean,t_mean,s_cov,t_cov] = mix_bernoulli_sample(N,P,Pi)

% Argument defaults
if nargin==2 Pi=1; P=P(1,:); end;   % Trivial mixture

[M,D] = size(P);
rand('seed',sum(100*clock));

% Block generation: generate exactly N*Pi(i) of each component
C = round(N*Pi);
if sum(C)~=N
  [i,j] = max(C);
  C(j) = C(j) + N - sum(C);
end

X = [];
for i=1:M
  X = [X; rand(C(i),D) <= ones(C(i),1)*P(i,:) ];
end

% $$$ % One-by-one generation: select randomly a component each time (slower)
% $$$ for i=1:M-2
% $$$   Pi(i+1) = Pi(i+1)+Pi(i);
% $$$ end
% $$$ Pi(M) = 1;
% $$$ X = zeros(N,D);
% $$$ for n=1:N
% $$$   % 1) Select component
% $$$   j = rand(1,1);
% $$$   i = 1;
% $$$   while j > Pi(i)
% $$$     i = i + 1;
% $$$   end
% $$$   % 2) Generate vector according to component i
% $$$   X(n,:) = rand(1,D) <= P(i,:);
% $$$ end

if nargout>1         % Compute experimental and theoretical mean vector
  if N==1 s_mean = X; else s_mean = sum(X)/N; end
  t_mean = Pi'*P;
end

if nargout>3         % Compute experimental and theoretical covariance matrix
  s_cov = X'*X/N - s_mean'*s_mean;
  t_cov = diag(Pi'*(P-P.^2)) + P'*(diag(Pi)-Pi*Pi')*P;
end
