% [reject,stat,chi,nu,maxllh] = fa_test(S,N,L[,maxllh]) Test for ML factor analysis
%
% Computes a likelihood ratio test statistic and, using the
% chi-squared, tells whether the null hipothesis:
%	H0: the data can be explained by *exactly* L factors
% should be rejected or not. The significance of the goodness-of-fit test
% is fixed to 95%.
%
% In:
%   S: DxD covariance matrix (>0).
%   N: number of data vectors (necessary to compute the log-likelihood).
%   L: number of factors (>=0).
%   maxllh: maximum log-likelihood. If omitted, it will be calculated by
%      the EM method to a relative error of 0.0001.
% Out:
%   reject: 1 if H0 is rejected, 0 otherwise. reject = (stat > chi). A -1
%       value is returned if the resulting number of degrees of freedom is
%       not positive.
%   stat: the likelihood ratio test statistic numerical value.
%   chi: the chi-squared critical point.
%   nu: number of degrees of freedom of the chi-squared.
%   maxllh: maximum log-likelihood. If omitted, it will be calculated by
%      the EM method to a relative error of 1e-9.
%
% See also fa2, chi2inv.

% Copyright (c) 1997 by Miguel A. Carreira-Perpinan

function [reject,stat,chi,nu,maxllh] = fa_test(S,N,L,maxllh)

[D,D] = size(S);

nu = ((D-L)^2-(D+L))/2;	% Degrees of freedom of the chi-square

if nu<=0
  reject = -1;
else
  % Argument defaults
  if nargin==3
    [lambda,phi,scores,maxllh] = fa2(S,N,L,1,3,1,1e-9);
    maxllh = maxllh(length(maxllh));
  end;
  chi = chi2inv(0.95,nu);
  stat = - ( N-1-(2*D+5)/6-2*L/3 ) * ( 2*maxllh/N + log(det(S)) + ...
             D*(1+log(2*pi)) );
  reject = stat > chi;
end
