% [A,Rf] = fa_scores(S,lambda,s,phi) Factor analysis scores and correlations
%
% In:
%   S: DxD (sample) covariance matrix (>0) of some data set.
%   lambda: DxL matrix of factor loadings in L dimensions obtained by
%      factor analysis.
%   s: scoring method (0: pseudoinverse; 1: Thomson scores; 2:
%      Bartlett scores). Default 1.
%   phi: D-vector (>0) containing the uniquenesses.
%
% Out:
%   A: LxD score matrix.
%   Rf: factor correlations. Rf(i,j) is the correlation between
%      factors i and j wrt S and the desired scoring.
%
% Notes:
%   1) Rf is invariant under scaling of the directions
%      (lambda2 = lambda*D with D diagonal nonsingular)
%      except for Thomson scores.
%   2) If lambda is the identity, Rf will be the standard correlation
%      matrix for S.
%
% See also: fa, fa_corr.

% Copyright (c) 1997 by Miguel A. Carreira-Perpinan

function [A,Rf] = fa_scores(S,lambda,s,phi)

% Argument defaults
if nargin==2 s=1; end;

[D,L] = size(lambda);

% Compute score matrix A
if s==0
  A = pinv(lambda);
elseif s==1
  phi_i = diag(1./phi);
  A = inv(eye(L)+lambda'*phi_i*lambda) * lambda' * phi_i;
elseif s==2
  phi_i = diag(1./phi);
  A = inv(lambda'*phi_i*lambda) * lambda' * phi_i;
end

% Compute factor correlation matrix Rf
if nargout == 2
  Rf = fa_corr(A,S);
end

