% [lambda,phi] = fa_guess(S,L,g) Starting point for ML factor analysis
%
% Provides with a starting point for maximum likelihood factor analysis.
%
% In:
%   S: DxD covariance matrix.
%   L: number of factors.
%   g: one of (default 3):
%      0: principal component solution.
%      1: principal component with structure removal (Maxwell 1964).
%      2: random (scaled) point.
%      3: LISREL (Joreskog 1969): phi = (1-L/2D) 1./diag(inv(S)).
% Out:
%   lambda: DxL matrix of loadings.
%   phi: Dx1 vector of uniquenesses.
%
% See also fa.

% Copyright (c) 1997 by Miguel A. Carreira-Perpinan

function [lambda,phi] = fa_guess(S,L,g)

% Argument defaults
if nargin==2 g=3; end;

[D,D] = size(S);

if g==0		% Principal component solution
  [A,J,A] = svd(S);
  lambda = A(:,1:L) * sqrt(J(1:L,1:L));
  phi = diag(S-lambda*lambda');
elseif g==1	% The loadings for the Lth factor is the principal component
		% of what is left from the rest (Maxwell 1964)
  S2 = S; lambda = [];
  for i=1:L
    [A,J,A] = svd(S2);
    lambda = [lambda A(:,1)];
    S2 = S2 - A(:,1)*A(:,1)';
  end
  phi = diag(S-lambda*lambda');
  if any(phi<=0)
    phi = diag(S);
  end
elseif g==2	% Random loadings and uniquenesses. lambda is scaled so
		% that trace(lambda*lambda') is of the order of trace(S)
  randn('seed',sum(100*clock));
  lambda = randn(D,L)*sqrt(trace(S)/D/L);
  phi = diag(S);
else		% Used by Joreskog in LISREL. Gives the best results
  phi = (1-L/2/D) ./ diag(inv(S));
  [A,J,A] = svd(S-diag(phi));
  lambda = A(:,1:L) * sqrt(J(1:L,1:L));
end
