% [lambda,phi,scores,m,S,llh,Sigma,code,i,gradn,error2] = fa(X,L[,me,g,tol,max_it,lambda0,phi0])
%
% Computes maximum likelihood factor analysis by one of these methods:
%
% - Rao (1953). Reference: D. Morrison: Multivariate Statistical Methods
%   3rd ed., pp. 357-362. McGraw-Hill, 1990.
%
% - EM algorithm (Rubin and Thayer 1982).
%
% In:
%   X: NxD data matrix of N D-dimensional row vectors.
%   L: number of factors (>=0).
%   me: method used (0: Rao, 1: EM; default 1).
%   g: starting point for the iterative procedure (0: principal component
%      solution, 1: principal component with structure removal, 2: random
%      scaled, 3: LISREL guess, 4: provided in lambda0, phi0; default 3).
%   s: factor scores (0: pseudoinverse, 1: Thomson, 2: Bartlett).
%      Default 1.
%   tol: minimum relative increase in log-likelihood to keep iterating
%      (default 1e-7 = 7 decimal places).
%   max_it: maximum number of iterations (default 400).
%   lambda0: initial loadings (if s=4).
%   phi0: initial uniquenesses (if s=4).
% Out:
%   lambda: DxL matrix of loadings.
%   phi: Dx1 vector of uniquenesses.
%   scores: LxD factor score matrix.
%   m: Dx1 vector with the sample mean.
%   S: DxD sample covariance matrix.
%   llh: log-likelihood curve of the parameters lambda and phi given the
%      data as a function of the iteration number.
%   Sigma: DxD symmetric matrix containing the model covariance matrix 
%      for factor analysis, i.e. lambda*lambda'+diag(phi). This is
%      useful to compute the log-likelihood under the same model of a
%      different data set using the function llh_normal.
%   code: stopping reason (0: tolerance achieved, 1: maximum number of
%      iterations reached, 2: Heywood case; 3: log-likelihood decreases
%      during learning, probably due to precision loss near maximum).
%   i: iterations performed.
%   gradn: square root of the average squared value of the log-likelihood
%      gradient.
%   error2: quadratic average sample error for factor scores.
%
% See also fa2, fa_scores, fa_guess, lin_proj_error, llh_normal.

% Copyright (c) 1997 by Miguel A. Carreira-Perpinan
%
% Ghahramani's ffa.m function performs exactly the same EM algorithm but:
%   1) He uses a random starting point.
%   2) He implements the EM iteration in a different way to make it
%      faster (my EM iteration takes 1.5 times the time his takes).

function [lambda,phi,scores,m,S,llh,Sigma,code,i,gradn,error2] = fa(X,L,me,g,s,tol,max_it,lambda0,phi0)

% Argument defaults
if nargin==2 me=1; end;
if nargin<=3 g=3; end;
if nargin<=4 s=1; end;
if nargin<=5 tol=1e-7; end;
if nargin<=6 max_it=400; end;

[N,D] = size(X);
m = mean(X);                   % Sample mean
X = X - ones(N,1)*m;           % Centre X
S = X'*X/N;                    % Sample covariance

if nargin >=9
  [lambda,phi,scores,llh,Sigma,code,i,gradn] = ...
      fa2(S,N,L,me,g,s,tol,max_it,lambda0,phi0);
else
    [lambda,phi,scores,llh,Sigma,code,i,gradn] = fa2(S,N,L,me,g,s,tol,max_it);
end

if nargout > 10
  error2 = lin_proj_error(X,scores',lambda);
end

