% matrix2ps(M,file[,grid[,norm[,scope[,threshold[,per]) Matrix to B&W EPS file
%
% Creates an Encapsulated PostScript File (EPSF) of the matrix M, where
% each matrix element is represented by a square of size proportional to
% its absolute value and colour white or black depending whether it is
% negative or positive (this is sometimes called a "Hinton diagram").
%
% M: input matrix.
% file: output file name; an extension .eps is added.
% grid: if different from 0, plot a grid.
% norm: one of 0,1,2,3,4; see lin_normalise for an explanation (default 1).
% scope: one of 0, 1 2 (default 0 for global normalisation).
% threshold in [0,1): minimum pixel square side to be plotted (default 0).
% per in (0,1]: relative space available for each pixel square (default 0.9).
%
% See also image_matrix, lin_normalise, u_colormap, matrix2pnm.

% Copyright (c) 1997 by Miguel A. Carreira-Perpinan

function matrix2ps(M,file,norm,scope,threshold,per)

% Argument defaults
if nargin==2 grid=0; end
if nargin==3 norm=1; end
if nargin<=4 scope=0; end
if nargin<=5 threshold=0; end;
if nargin<=6 per=0.9; end

clf reset;
[m,n]=size(M);
M = lin_normalise(M,norm,scope);
hold on;

for i=1:m
  for j=1:n
    a=abs(M(i,j))*per/2;	% Square side
    if a>threshold		% Plot square only if it is not too small
      if M(i,j)<0		% Empty square
        line([j-a j+a j+a j-a j-a]',...
             [n+1-i-a n+1-i-a n+1-i+a n+1-i+a n+1-i-a]','Color','w');
      else			% Black-filled square
        fill([j-a j+a j+a j-a],[n+1-i-a n+1-i-a n+1-i+a n+1-i+a],'w');
      end
    end
  end
end

if grid==0
  % Invisible (white) line to get the correct bounding box in the EPS file
  line([0.5 m+0.5 m+0.5 0.5 0.5]',[n+0.5 n+0.5 0.5 0.5 n+0.5]','Color','k');
else
  for i=1:m+1
    line([0.5 n+0.5]',[i-0.5 i-0.5]','Color','w');
  end
  for j=1:n+1
    line([j-0.5 j-0.5]',[0.5 m+0.5]','Color','w');
  end
end

axis([0.5 m+0.5 0.5 n+0.5]); axis('square'); axis('off');

hold off;

eval(['print -deps ',file,'.eps']);
