% matrix2pnm(M,file[,cmap[,norm[,scope]) Matrix to PNM bitmap file
%
% Transforms a matrix M into a pgm (greyscale) or ppm (colour) file
% with the colormap and normalisation specified.
%
% file: output file name. Suffix '.pgm' or '.ppm' is added correspondingly.
%
% cmap can be one of:
%   - 'g' for a greyscale colormap (black < white).
%   - 'u' for a U-shaped colormap (blue < black < red).
%
% scope can be 0, 1 or 2 and norm 0, 1, 2, 3 or 4 (see lin_normalise for
% an explanation).
%
% For data in [0,1] use norm=0 or 1 and cmap='g' or 'u'; for data in
% [-1,1] use norm=2 or 3 and cmap='u'. If uncertain, norm3 is the best
% choice: it will not lie about the data. Defaults: 'u' and 3 or 'g' and 1.
%
% To do: create colormaps that give a correct visual impression of the
% actual intensity of the pixel
%
% See also image_matrix, lin_normalise, u_colormap, matrix2ps.

% Copyright (c) 1997 by Miguel A. Carreira-Perpinan

function matrix2pnm(M,file,cmap,norm,scope)

% Argument defaults
if nargin==2 cmap='u'; end;
if nargin<=3 if cmap=='u' norm=3; else norm=1; end; end;
if nargin<=4 scope=0; end;

% Normalisation
M = lin_normalise(M,norm,scope);

% Convert to indices in a colormap of N colours
if cmap=='g'
  N=64;
  M = round((N-1)*M);
else
  N=255;
  palette=round(N*u_colorpal(N));
  M = round((N-1)*M+1);
end

% Image dimensions
[r c] = size(M);

% Write to pgm or ppm file
if cmap=='g'
  fid=fopen([file,'.pgm'],'w');
  fprintf(fid,'P2\n%d %d %d\n',c,r,N-1);
  for i=1:r
    for j=1:c
      fprintf(fid,'%d ',M(i,j));
    end
  end
  fclose(fid);
else
  fid=fopen([file,'.ppm'],'w');
  fprintf(fid,'P3\n%d %d %d\n',c,r,N);
  for i=1:r
    for j=1:c
      fprintf(fid,'%d ',palette(M(i,j),:));
    end
  end
  fclose(fid);
end
