% image_matrix(M[,cmap[,norm[,scope[,N]) Colour image of a matrix
%
% Plots to screen the image of matrix M with the colormap and
% normalisation specified.
%
% cmap can be one of:
%   - 'g' for a greyscale colormap (black < white).
%   - 'u' for a U-shaped colormap (blue < black < red).
%
% N is the number of colours in the colormap. If cmap='u' then N must
% be an odd integer (typically 2^n-1) to allow for the black colour in
% the colormap centre. Default: 64 for 'g' and 255 for 'u'.
%
% scope can be 0, 1 or 2 and norm 0, 1, 2, 3 or 4 (see lin_normalise for
% an explanation).
%
% For data in [0,1] use norm=0 or 1 and cmap='g' or 'u'; for data in
% [-1,1] use norm=2 or 3 and cmap='u'. If uncertain, norm3 is the best
% choice: it will not lie about the data. Defaults: 'u' and 3 or 'g' and 1.
%
% To do: create colormaps that give a correct visual impression of the
% actual intensity of the pixel
%
% See also lin_normalise, u_colormap, matrix2pnm, matrix2ps.

% Copyright (c) 1997 by Miguel A. Carreira-Perpinan

function h = image_matrix(M,cmap,norm,scope,N)

% Argument defaults
if nargin==1 cmap='u'; end;
if nargin<=2 if cmap=='u' norm=3; else norm=1; end; end;
if nargin<=3 scope=0; end;
if nargin<=4 if cmap=='u' N=255; else N=64; end; end;

% Colormap of N colours
if cmap=='u'
  if rem(N,2)==0
    N=N+1;
    fprintf(['Warning: U-shaped colormaps require an odd number of ', ...
            'colours.\nI will use %d colours.\n'],N);
  end
  colormap(u_colorpal(N));
else
  colormap(gray(N));
end

% Image
h = image( (N-1) * lin_normalise(M,norm,scope) );
axis('image');
axis('off');

