% image_epg_frame(f[,cmap[,norm[,N]) Colour image of an EPG frame
%
% Plots to screen an image of the epg vector f (with 62 components).
%
% cmap can be one of 'g','u' and norm one of 0,1,2,3,4 (see
% lin_normalise for an explanation). Defaults: 'u' and 3 or 'g' and 1.
%
% N is the number of colours in the colormap. If cmap='u' then N must
% be an odd integer (typically 2^n-1) to allow for the black colour in
% the colormap centre. Default: 64 for 'g' and 255 for 'u'.
%
% For data in [0,1] use norm=0 or 1 and cmap='g' or 'u'; for data in
% [-1,1] use norm=2 or 3 and cmap='u'. If uncertain, norm3 is the best
% choice: it will not lie about the data.
%
% See also image_matrix, lin_normalise, u_colormap, epg2pnm, epglist2pnm,
% epg2ps, epglist2ps.

% Copyright (c) 1997 by Miguel A. Carreira-Perpinan

function h = image_epg_frame(f,cmap,norm,N)

% Argument defaults
if nargin==1 cmap='u'; end;
if nargin<=2 if cmap=='u' norm=3; else norm=1; end; end;
if nargin<=3 if cmap=='u' N=255; else N=64; end; end;

% Normalisation
f = lin_normalise(f,norm);

% Colormap of N colours
if cmap=='u'
  if rem(N,2)==0
    N=N+1;
    fprintf(['Warning: U-shaped colormaps require an even number of ', ...
            'colours.\nI will use %d colours.\n'],N);
  end
  colormap(u_colorpal(N));
else
  colormap(gray(N));
end

% Create matrix out of vector and plot it
if cmap=='g'
  h = image( (N-1) * [0 f(1:6)' 0; f(7:14)'; f(15:22)'; f(23:30)'; f(31:38)'; f(39:46)'; f(47:54)'; f(55:62)'] );
else
  h = image( (N-1) * [((N-1)/2+1)/N f(1:6)' ((N-1)/2+1)/N; f(7:14)'; f(15:22)'; f(23:30)'; f(31:38)'; f(39:46)'; f(47:54)'; f(55:62)'] );
end
axis('image');
axis('off');

