% epglist2pscollage(l,file[,grid,bkg,squared_comp,labels,H,V,scope,norm])
% EPG frames to B&W EPS collage
%
% Plots a list of epg vectors as a black and white collage in EPS format.
%
% l: matrix of r x 62, containing r EPG vectors.
% file: file name for the output EPS file.
% grid: 0 (no grid, default), 1 (dotted, full grid), 2 (solid outline).
% bkg: real number in [0,1] indicating the background grey level, from
%    0=black to 1=white (default). bkg=0.8 prints nicely.
% squared_comp: 1 (squared electrodes, default); 0 (circular electrodes).
% labels: 0 (no labels, default), 1 (each frame is labelled with a
%    correlative number from 1 to r).
% H, V: number of subplots in horizontal and vertical, respectively (default
%    H=V with H*V>=r).
% scope: should be either 0 or 2; see lin_normalise (default 2=rowwise, i.e.
%    normalisation local to each vector).
% norm: see lin_normalise (default 1).
%
% See also epg2ps, epglist2ps, epg2pnm, epglist2pnm, image_epg_frame,
% image_matrix, matrix2pnm, matrix2ps, lin_normalise, u_colormap.

% Copyright (c) 1998 by Miguel A. Carreira-Perpinan

function epglist2pscollage(l,file,grid,bkg,squared_comp,labels,H,V,scope,norm)

[r c]=size(l);

% Argument defaults
if nargin==2 grid=0; end;
if nargin<=3 bkg=1; end;
if nargin<=4 squared_comp=1; end;
if nargin<=5 labels=0; end;
if nargin<=6 H=ceil(sqrt(r)); V=H; end;
if nargin<=7 V=ceil(r/H); end;
if nargin<=8 scope=2; end;
if nargin<=9 norm=1; end;

l = lin_normalise(l,norm,scope);	% Normalise here

threshold=0; per=0.9;

clf reset; set(gcf,'Color','w');

for count=1:r
  subplot(V,H,count);
  hold on;
  % Create sequence of pixel intensities out of vector
  f = l(count,:);
  M = [0 f(1:6) 0;f(7:14);f(15:22);f(23:30);f(31:38);f(39:46);f(47:54);f(55:62)];
  [m,n]=size(M);
  % Fill an EPG shape with the background colour (the last vertex of the
  % shape is not specified, as fill will close the polygon).
  % Creating this filled polygon also produces a correct bounding box in
  % the EPS file even when the background is white.
  fill([0.5 1.5 1.5 m-0.5 m-0.5 m+0.5 m+0.5 0.5]', ...
       [n-0.5 n-0.5 n+0.5 n+0.5 n-0.5 n-0.5 0.5 0.5]', ...
       bkg*[1 1 1],'EdgeColor',bkg*[1 1 1]);
  % Plot all 62 components
  if squared_comp==1		% Plot squared components
    for i=1:m
      for j=1:n
	a=abs(M(i,j))*per/2;	% Square half-side
	if a>threshold		% Plot square only if it is not too small
	  if M(i,j)<0		% White-filled square
	    fill([j-a j+a j+a j-a],[n+1-i-a n+1-i-a n+1-i+a n+1-i+a],'w');
	  else			% Black-filled square
	    fill([j-a j+a j+a j-a],[n+1-i-a n+1-i-a n+1-i+a n+1-i+a],'k');
	  end
	end
      end
    end
  else				% Plot circular components
    circ_num = 20;		% Number of points to construct the circles
    circle = [cos(0:2*pi/(circ_num-1):2*pi); sin(0:2*pi/(circ_num-1):2*pi)];
    for i=1:m
      for j=1:n
	a=abs(M(i,j))*per/2;	% Radius
	if a>threshold		% Plot circle only if it is not too small
	  circ=circle*a+[j; n+1-i]*ones(1,circ_num);
	  if M(i,j)<0		% White-filled circle
	    fill(circ(1,:),circ(2,:),'w');
	  else			% Black-filled circle
	    fill(circ(1,:),circ(2,:),'k');
	  end
	end
      end
    end
  end
  % Fancy grid
  if grid == 1
    % Full grid
    for i=1:m
      line([0.5 n+0.5]',[i-0.5 i-0.5]','Color','k','LineStyle',':');
    end
    for j=2:n
      line([j-0.5 j-0.5]',[0.5 m+0.5]','Color','k','LineStyle',':');
    end
    line([1.5 n-0.5]',[m+0.5 m+0.5]','Color','k','LineStyle',':');
    line([0.5 0.5]',[0.5 m-0.5]','Color','k','LineStyle',':');
    line([n+0.5 n+0.5]',[0.5 m-0.5]','Color','k','LineStyle',':');
  elseif grid==2
    % Outline
    line([0.5 1.5 1.5 m-0.5 m-0.5 m+0.5 m+0.5 0.5 0.5]',[n-0.5 n-0.5 ...
	n+0.5 n+0.5 n-0.5 n-0.5 0.5 0.5 n-0.5]','Color','k');
  end
  axis([0.5 m+0.5 0.5 n+0.5]); axis('square'); axis('off');
  if labels~=0
    text(m/2+0.5,0,int2str(count));
  end
end

hold off;

eval(['print -deps ',file,'.eps']);
