% N = lin_normalise(M,norm[,scope]) Matrix linear normalisation
%
% Normalises matrix M in a linear way. Ideally (i.e. choosing the
% correct norm for the particular data in M) all values are mapped to
% the [0,1] interval, so that its components can be matched against a
% colormap by multiplying by an integer (the number of colours in the
% colormap minus one) and rounding to integer and thus obtain a colour
% image.
%
% scope can be 0 for global, 1 for columnwise or 2 for rowwise.
% Default is global.
%
% norm can be one of (where cmap='g' means a grey shades colormap and
% cmap='u' means a U-shaped colormap, starting from blue, passing
% through black and ending in red):
%   - 0: N = M with cmap='g', useful for data in [0,1].
%   - 1: N = M/max|M| with cmap='g', useful for data in [0,inf] or to
%        brighten norm0.
%   - 2: N = (M+1)/2 with cmap='u', useful for data in [-1,1]: blue<0,
%        black=0, red>0. Also works with data in [0,1]: black=0, red>0;
%        or [-1,0]: blue<0, black=0.
%   - 3: N = (M/max|M|+1)/2 with cmap='u', useful for data in [-inf,inf]
%        or to brighten norm2: blue<0, black=0, red>0. Also works with
%        data in [0,inf]: black=0, red>0; or [-inf,0]: blue<0, black=0.
%   - 4: N = (M-min(M))/(max(M)-min(M)) with cmap='g', useful for data
%        in [-inf,inf].
%
% For data in [0,1] use norm0-1; for data in [-1,1] use norm2-3.
% If uncertain, norm3 is the best choice: it will not lie about the data.
%
% See also image_matrix, u_colormap, matrix2pnm, matrix2ps.

% Copyright (c) 1997 by Miguel A. Carreira-Perpinan

function N = lin_normalise(M,norm,scope)

% Argument defaults
if nargin==2 scope=0; end;

if norm==0
  N = M;
elseif norm==1
  if scope==0
    N = M / max(max(abs(M)));
  elseif scope==1
    N = M * diag(1./max(abs(M)));
  elseif scope==2
    N = diag(1./max(abs(M'))) * M;
  end
elseif norm==2
  N = (M+1)/2;
elseif norm==3
  if scope==0
    N = (M / max(max(abs(M))) +1)/2;
  elseif scope==1
    N = (M * diag(1./max(abs(M))) +1)/2;
  elseif scope==2
    N = (diag(1./max(abs(M'))) * M +1)/2;
  end
elseif norm==4
  [m,n] = size(M);
  if scope==0
    N = (M-min(min(M)))/(max(max(M))-min(min(M)));
  elseif scope==1
    N = (M-ones(m,1)*min(M)) * diag(1./(max(M)-min(M)));
  elseif scope==2
    N = diag(1./(max(M')-min(M'))) * (M-(min(M'))'*ones(1,n));
  end
end

