% epg2ps(f,file[,grid,bkg,squared_comp,norm]) EPG frame to B&W EPS file
%
% Transforms an epg vector into a black and white Encapsulated
% PostScript File (EPSF) where each element is represented by a square
% of size proportional to its absolute value and colour white or black
% depending whether it is negative or positive.
%
% f: 62-vector, to be transformed to EPS.
% file: output file name. Suffix '.eps' is added automatically.
% grid: 0 (no grid, default), 1 (dotted, full grid), 2 (solid outline).
% bkg: real number in [0,1] indicating the background grey level, from
%    0=black to 1=white (default). bkg=0.8 prints nicely.
% squared_comp: 1 (squared electrodes, default); 0 (circular electrodes).
% norm: one of 0,1,2,3,4; see lin_normalise for an explanation (default 1).
%
% See also epglist2ps, epglist2pscollage, epg2pnm, epglist2pnm,
% image_epg_frame, image_matrix, matrix2pnm, matrix2ps, lin_normalise,
% u_colormap.

% Copyright (c) 1998 by Miguel A. Carreira-Perpinan

function epg2pnm(f,file,grid,bkg,squared_comp,norm)

% Argument defaults
if nargin==2 grid=0; end;
if nargin<=3 bkg=1; end;
if nargin<=4 squared_comp=1; end;
if nargin<=5 norm=1; end;

% Normalisation
f = lin_normalise(f,norm,0);

% Create sequence of pixel intensities out of vector
M = [0 f(1:6)' 0;f(7:14)';f(15:22)';f(23:30)';f(31:38)';f(39:46)';f(47:54)';f(55:62)'];

% This is equivalent to matrix2ps(M,file,1,0) but with a fancier grid

threshold=0; per=0.9;

clf reset; set(gcf,'Color','w');
[m,n]=size(M);
hold on;

% Fill an EPG shape with the background colour (the last vertex of the
% shape is not specified, as fill will close the polygon).
% Creating this filled polygon also produces a correct bounding box in
% the EPS file even when the background is white.
fill([0.5 1.5 1.5 m-0.5 m-0.5 m+0.5 m+0.5 0.5]', ...
     [n-0.5 n-0.5 n+0.5 n+0.5 n-0.5 n-0.5 0.5 0.5]', ...
     bkg*[1 1 1],'EdgeColor',bkg*[1 1 1]);

% Plot all 62 components
if squared_comp==1		% Plot squared components
  for i=1:m
    for j=1:n
      a=abs(M(i,j))*per/2;	% Square half-side
      if a>threshold		% Plot square only if it is not too small
	if M(i,j)<0		% White-filled square
	  fill([j-a j+a j+a j-a],[n+1-i-a n+1-i-a n+1-i+a n+1-i+a],'w');
	else			% Black-filled square
	  fill([j-a j+a j+a j-a],[n+1-i-a n+1-i-a n+1-i+a n+1-i+a],'k');
	end
      end
    end
  end
else				% Plot circular components
  circ_num = 20;		% Number of points to construct the circles
  circle = [cos(0:2*pi/(circ_num-1):2*pi); sin(0:2*pi/(circ_num-1):2*pi)];
  for i=1:m
    for j=1:n
      a=abs(M(i,j))*per/2;	% Radius
      if a>threshold		% Plot circle only if it is not too small
	circ=circle*a+[j; n+1-i]*ones(1,circ_num);
	if M(i,j)<0		% White-filled circle
	  fill(circ(1,:),circ(2,:),'w');
	else			% Black-filled circle
	  fill(circ(1,:),circ(2,:),'k');
	end
      end
    end
  end
end

% Fancy grid
if grid == 1
  % Full grid
  for i=1:m
    line([0.5 n+0.5]',[i-0.5 i-0.5]','Color','k','LineStyle',':');
  end
  for j=2:n
    line([j-0.5 j-0.5]',[0.5 m+0.5]','Color','k','LineStyle',':');
  end
  line([1.5 n-0.5]',[m+0.5 m+0.5]','Color','k','LineStyle',':');
  line([0.5 0.5]',[0.5 m-0.5]','Color','k','LineStyle',':');
  line([n+0.5 n+0.5]',[0.5 m-0.5]','Color','k','LineStyle',':');
elseif grid==2
  % Outline
  line([0.5 1.5 1.5 m-0.5 m-0.5 m+0.5 m+0.5 0.5 0.5]',[n-0.5 n-0.5 ...
      n+0.5 n+0.5 n-0.5 n-0.5 0.5 0.5 n-0.5]','Color','k');
end

axis([0.5 m+0.5 0.5 n+0.5]); axis('square'); axis('off');

hold off;

eval(['print -deps ',file,'.eps']);
