% epg2pnm(f,file[,cmap[,norm]) EPG frame to PNM bitmap file
%
% Transforms an epg vector into a pgm (greyscale) or ppm (colour) file.
%
% f: 62-vector, to be transformed to 8x16 pgm (ppm) ascii image.
% file: output file name. Suffix '.pgm' or '.ppm' is added correspondingly.
%
% cmap can be one of 'g','u' and norm one of 0,1,2,3,4 (see image_epg_frame
% for an explanation). Defaults: 'u' and 3 or 'g' and 1.
%
% For data in [0,1] use norm=0 or 1 and cmap='g' or 'u'; for data in
% [-1,1] use norm=2 or 3 and cmap='u'. If uncertain, norm3 is the best
% choice: it will not lie about the data.
%
% See also epglist2pnm, epg2ps, epglist2ps, epglist2pscollage,
% image_epg_frame, image_matrix, matrix2pnm, matrix2ps, lin_normalise,
% u_colormap.

% Copyright (c) 1997 by Miguel A. Carreira-Perpinan

function epg2pnm(f,file,cmap,norm)

% Argument defaults
if nargin==2 cmap='u'; end;
if nargin<=3 if cmap=='u' norm=3; else norm=1; end; end;

% Normalisation
f = lin_normalise(f,norm,0);

% Convert to indices in a colormap of N colours
if cmap=='g'
  N=64;
  f = round((N-1)*f);
else
  N=255;
  palette=round(N*u_colorpal(N));
  f = round((N-1)*f+1);
end

% Create sequence of pixel intensities out of vector; each row is duplicated
% so that the aspect ratio is 1:2, which is nicer to look at.
if cmap=='g'
  m = [N-1 f(1:6)' N-1 N-1 f(1:6)' N-1 f(7:14)' f(7:14)' f(15:22)' f(15:22)' f(23:30)' f(23:30)' f(31:38)' f(31:38)' f(39:46)' f(39:46)' f(47:54)' f(47:54)' f(55:62)' f(55:62)'];
else
  m = [(N-1)/2+1 f(1:6)' (N-1)/2+1 (N-1)/2+1 f(1:6)' (N-1)/2+1 f(7:14)' f(7:14)' f(15:22)' f(15:22)' f(23:30)' f(23:30)' f(31:38)' f(31:38)' f(39:46)' f(39:46)' f(47:54)' f(47:54)' f(55:62)' f(55:62)'];
end

% Write to pgm or ppm file
if cmap=='g'
  fid=fopen([file,'.pgm'],'w');
  fprintf(fid,'P2\n8 16 %d\n',N-1);
  fprintf(fid,'%d ',m);
  fclose(fid);
else
  fid=fopen([file,'.ppm'],'w');
  fprintf(fid,'P3\n8 16 %d\n',N);
  for i=1:length(m)
    fprintf(fid,'%d ',palette(m(i),:));
  end
  fclose(fid);
end
